// 本机设置功能

// 切换设置标签页
function switchSettingsTab(tab) {
    const pageSettings = document.getElementById('page-localsettings');
    if (!pageSettings) return;
    
    // 隐藏所有标签页内容
    const tabContents = pageSettings.querySelectorAll('.settings-tab-content');
    tabContents.forEach(el => {
        el.style.display = 'none';
        el.classList.remove('active');
    });
    
    // 重置所有按钮样式
    const buttons = pageSettings.querySelectorAll('.tab-btn');
    buttons.forEach(btn => {
        btn.classList.remove('active');
    });
    
    // 显示选中的标签页
    const targetTab = document.getElementById(`settings-tab-${tab}`);
    if (targetTab) {
        targetTab.style.display = 'block';
        targetTab.classList.add('active');
    }
    
    // 高亮选中的按钮
    buttons.forEach(btn => {
        const btnText = btn.textContent.trim();
        const tabNames = {
            'network': '网络配置',
            'system': '系统设置',
            'services': '系统服务',
            'firewall': '防火墙',
            'sysctl': '内核参数'
        };
        if (btnText === tabNames[tab]) {
            btn.classList.add('active');
        }
    });
    
    // 根据标签页加载数据
    if (tab === 'network') {
        refreshInterfaces();
    } else if (tab === 'system') {
        loadSystemInfo();
        loadHostname();
        loadTimezone();
    } else if (tab === 'services') {
        refreshServices();
    } else if (tab === 'firewall') {
        refreshFirewallStatus();
    } else if (tab === 'sysctl') {
        refreshSysctlParams();
    }
}

// ==================== 网络配置 ====================

async function refreshInterfaces() {
    const container = document.getElementById('interfaces-table-container');
    const select = document.getElementById('settings-interface-select');
    
    if (container) container.innerHTML = '加载中...';
    
    try {
        const response = await fetch('/api/settings/interfaces');
        const data = await response.json();
        
        if (data.success && data.interfaces) {
            // 更新下拉选择
            if (select) {
                select.innerHTML = '<option value="">请选择接口...</option>';
                data.interfaces.forEach(iface => {
                    const option = document.createElement('option');
                    option.value = iface.name;
                    option.textContent = `${iface.name} (${iface.state})`;
                    select.appendChild(option);
                });
            }
            
            // 更新接口列表
            if (container) {
                let html = '<table class="result-table"><thead><tr><th>接口名</th><th>状态</th><th>IPv4</th><th>IPv6</th><th>MAC地址</th></tr></thead><tbody>';
                data.interfaces.forEach(iface => {
                    html += `<tr>
                        <td><strong>${iface.name}</strong></td>
                        <td><span class="interface-status ${iface.is_up ? 'up' : 'down'}">${iface.state}</span></td>
                        <td>${iface.ipv4}</td>
                        <td>${iface.ipv6}</td>
                        <td>${iface.mac}</td>
                    </tr>`;
                });
                html += '</tbody></table>';
                container.innerHTML = html;
            }
        } else {
            if (container) container.innerHTML = `❌ ${data.error || '获取接口列表失败'}`;
        }
    } catch (error) {
        if (container) container.innerHTML = `❌ 请求失败: ${error.message}`;
    }
}

async function loadInterfaceConfig() {
    const select = document.getElementById('settings-interface-select');
    const details = document.getElementById('interface-config-details');
    if (!select || !details) return;
    
    const interface = select.value;
    if (!interface) {
        details.style.display = 'none';
        return;
    }
    
    try {
        const response = await fetch(`/api/settings/interface/${interface}`);
        const data = await response.json();
        
        if (data.success && data.config) {
            const config = data.config;
            document.getElementById('settings-config-method').value = config.method || 'dhcp';
            document.getElementById('settings-ip').value = config.ipv4 || '';
            document.getElementById('settings-netmask').value = config.netmask || '';
            document.getElementById('settings-gateway').value = config.gateway || '';
            document.getElementById('settings-dns').value = config.dns.join(',') || '';
            
            toggleStaticConfig();
            details.style.display = 'block';
        } else {
            alert(`获取配置失败: ${data.error || '未知错误'}`);
        }
    } catch (error) {
        alert(`请求失败: ${error.message}`);
    }
}

function toggleStaticConfig() {
    const method = document.getElementById('settings-config-method').value;
    const staticConfig = document.getElementById('static-config');
    if (staticConfig) {
        staticConfig.style.display = method === 'static' ? 'block' : 'none';
    }
}

async function saveInterfaceConfig() {
    alert('⚠️ 网络配置保存功能需要root权限，且需要修改netplan配置文件。\n此功能建议通过命令行手动配置。');
    // TODO: 实现netplan配置文件的生成和写入
}

function resetInterfaceConfig() {
    document.getElementById('settings-ip').value = '';
    document.getElementById('settings-netmask').value = '';
    document.getElementById('settings-gateway').value = '';
    document.getElementById('settings-dns').value = '';
}

// ==================== 系统设置 ====================

async function loadSystemInfo() {
    const resultEl = document.getElementById('system-info-text');
    if (!resultEl) return;
    
    try {
        const response = await fetch('/api/settings/system/info');
        const data = await response.json();
        
        if (data.success && data.info) {
            const info = data.info;
            let output = `操作系统: ${info.os || 'N/A'}\n`;
            output += `发行版: ${info.distro || 'N/A'}\n`;
            output += `内核版本: ${info.os_release || 'N/A'}\n`;
            output += `主机名: ${info.hostname || 'N/A'}\n`;
            output += `架构: ${info.machine || 'N/A'}\n`;
            output += `处理器: ${info.processor || 'N/A'}\n`;
            output += `CPU核心数: ${info.cpu_count || 'N/A'}\n`;
            output += `总内存: ${(info.memory_total / 1024 / 1024 / 1024).toFixed(2)} GB\n`;
            output += `总磁盘: ${(info.disk_total / 1024 / 1024 / 1024).toFixed(2)} GB\n`;
            resultEl.textContent = output;
        } else {
            resultEl.textContent = `❌ ${data.error || '获取系统信息失败'}`;
        }
    } catch (error) {
        resultEl.textContent = `❌ 请求失败: ${error.message}`;
    }
}

async function loadHostname() {
    try {
        const response = await fetch('/api/settings/hostname');
        const data = await response.json();
        if (data.success && data.hostname) {
            const input = document.getElementById('settings-hostname');
            if (input) input.value = data.hostname;
        }
    } catch (error) {
        console.error('加载主机名失败:', error);
    }
}

async function saveHostname() {
    const input = document.getElementById('settings-hostname');
    if (!input) return;
    
    const hostname = input.value.trim();
    if (!hostname) {
        alert('主机名不能为空');
        return;
    }
    
    try {
        const response = await fetch('/api/settings/hostname', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({hostname})
        });
        const data = await response.json();
        
        if (data.success) {
            alert(`✅ ${data.message || '主机名设置成功'}`);
        } else {
            alert(`❌ ${data.error || '设置失败'}`);
        }
    } catch (error) {
        alert(`❌ 请求失败: ${error.message}`);
    }
}

async function loadTimezone() {
    try {
        const response = await fetch('/api/settings/timezone');
        const data = await response.json();
        if (data.success && data.timezone) {
            const select = document.getElementById('settings-timezone');
            if (select) {
                select.value = data.timezone;
            }
        }
    } catch (error) {
        console.error('加载时区失败:', error);
    }
}

async function saveTimezone() {
    const select = document.getElementById('settings-timezone');
    if (!select) return;
    
    const timezone = select.value;
    if (!timezone) {
        alert('请选择时区');
        return;
    }
    
    try {
        const response = await fetch('/api/settings/timezone', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({timezone})
        });
        const data = await response.json();
        
        if (data.success) {
            alert(`✅ ${data.message || '时区设置成功'}`);
        } else {
            alert(`❌ ${data.error || '设置失败'}`);
        }
    } catch (error) {
        alert(`❌ 请求失败: ${error.message}`);
    }
}

// ==================== 系统服务 ====================

let allServices = [];

async function refreshServices() {
    const container = document.getElementById('services-table-container');
    if (!container) return;
    
    container.innerHTML = '加载中...';
    
    try {
        const response = await fetch('/api/settings/services');
        const data = await response.json();
        
        if (data.success && data.services) {
            allServices = data.services;
            displayServices(allServices);
        } else {
            container.innerHTML = `❌ ${data.error || '获取服务列表失败'}`;
        }
    } catch (error) {
        container.innerHTML = `❌ 请求失败: ${error.message}`;
    }
}

function displayServices(services) {
    const container = document.getElementById('services-table-container');
    if (!container) return;
    
    if (services.length === 0) {
        container.innerHTML = '没有找到服务';
        return;
    }
    
    let html = '<table class="result-table"><thead><tr><th>服务名</th><th>状态</th><th>操作</th></tr></thead><tbody>';
    services.forEach(service => {
        const statusClass = service.is_running ? 'success' : 'danger';
        const statusText = service.is_running ? '运行中' : '已停止';
        html += `<tr>
            <td><strong>${service.name}</strong></td>
            <td><span class="btn btn-${statusClass} btn-small">${statusText}</span></td>
            <td>
                <button class="btn btn-primary btn-small" onclick="controlService('${service.name}', 'start')">启动</button>
                <button class="btn btn-danger btn-small" onclick="controlService('${service.name}', 'stop')">停止</button>
                <button class="btn btn-default btn-small" onclick="controlService('${service.name}', 'restart')">重启</button>
            </td>
        </tr>`;
    });
    html += '</tbody></table>';
    container.innerHTML = html;
}

function filterServices() {
    const search = document.getElementById('service-search').value.toLowerCase();
    const filtered = allServices.filter(s => s.name.toLowerCase().includes(search));
    displayServices(filtered);
}

async function controlService(serviceName, action) {
    if (!confirm(`确定要${action === 'start' ? '启动' : action === 'stop' ? '停止' : '重启'}服务 ${serviceName} 吗？`)) {
        return;
    }
    
    try {
        const response = await fetch(`/api/settings/service/${serviceName}/${action}`, {
            method: 'POST'
        });
        const data = await response.json();
        
        if (data.success) {
            alert(`✅ ${data.message || '操作成功'}`);
            refreshServices();
        } else {
            alert(`❌ ${data.error || '操作失败'}`);
        }
    } catch (error) {
        alert(`❌ 请求失败: ${error.message}`);
    }
}

// ==================== 防火墙 ====================

async function refreshFirewallStatus() {
    const resultEl = document.getElementById('firewall-status-text');
    if (!resultEl) return;
    
    try {
        const response = await fetch('/api/settings/firewall');
        const data = await response.json();
        
        if (data.success && data.status) {
            const status = data.status;
            let output = '防火墙状态:\n';
            output += '='.repeat(50) + '\n\n';
            output += `UFW: ${status.ufw.active ? '✅ 运行中' : '❌ 未运行'} (${status.ufw.enabled ? '已启用' : '未启用'})\n`;
            output += `Firewalld: ${status.firewalld.active ? '✅ 运行中' : '❌ 未运行'} (${status.firewalld.enabled ? '已启用' : '未启用'})\n`;
            output += `iptables: ${status.iptables.active ? '✅ 已配置' : '❌ 未配置'}\n`;
            resultEl.textContent = output;
        } else {
            resultEl.textContent = `❌ ${data.error || '获取防火墙状态失败'}`;
        }
    } catch (error) {
        resultEl.textContent = `❌ 请求失败: ${error.message}`;
    }
}

// ==================== 内核参数 ====================

async function refreshSysctlParams() {
    const container = document.getElementById('sysctl-table-container');
    if (!container) return;
    
    container.innerHTML = '加载中...';
    
    try {
        const response = await fetch('/api/settings/sysctl');
        const data = await response.json();
        
        if (data.success && data.params) {
            const params = data.params;
            if (Object.keys(params).length === 0) {
                container.innerHTML = '没有找到内核参数';
                return;
            }
            
            let html = '<table class="result-table"><thead><tr><th>参数名</th><th>当前值</th><th>操作</th></tr></thead><tbody>';
            for (const [param, value] of Object.entries(params)) {
                html += `<tr>
                    <td><strong>${param}</strong></td>
                    <td>${value}</td>
                    <td>
                        <button class="btn btn-primary btn-small" onclick="editSysctlParam('${param}', '${value}')">编辑</button>
                    </td>
                </tr>`;
            }
            html += '</tbody></table>';
            container.innerHTML = html;
        } else {
            container.innerHTML = `❌ ${data.error || '获取内核参数失败'}`;
        }
    } catch (error) {
        container.innerHTML = `❌ 请求失败: ${error.message}`;
    }
}

function editSysctlParam(param, currentValue) {
    const newValue = prompt(`编辑参数: ${param}\n当前值: ${currentValue}\n\n请输入新值:`, currentValue);
    if (newValue === null || newValue === currentValue) {
        return;
    }
    
    fetch('/api/settings/sysctl', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({param, value: newValue})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            alert(`✅ ${data.message || '参数设置成功'}`);
            refreshSysctlParams();
        } else {
            alert(`❌ ${data.error || '设置失败'}`);
        }
    })
    .catch(error => {
        alert(`❌ 请求失败: ${error.message}`);
    });
}

// 导出函数到全局作用域
window.switchSettingsTab = switchSettingsTab;
window.refreshInterfaces = refreshInterfaces;
window.loadInterfaceConfig = loadInterfaceConfig;
window.toggleStaticConfig = toggleStaticConfig;
window.saveInterfaceConfig = saveInterfaceConfig;
window.resetInterfaceConfig = resetInterfaceConfig;
window.loadSystemInfo = loadSystemInfo;
window.loadHostname = loadHostname;
window.saveHostname = saveHostname;
window.loadTimezone = loadTimezone;
window.saveTimezone = saveTimezone;
window.refreshServices = refreshServices;
window.filterServices = filterServices;
window.controlService = controlService;
window.refreshFirewallStatus = refreshFirewallStatus;
window.refreshSysctlParams = refreshSysctlParams;
window.editSysctlParam = editSysctlParam;

