let dashboardInterval = null;

document.addEventListener('DOMContentLoaded', function() {
    // 立即初始化导航（必须立即执行）
    initNavigation();
    
    // 使用requestAnimationFrame延迟其他初始化，避免阻塞页面渲染
    requestAnimationFrame(() => {
        initDashboard();
        loadInterfaces();
        
        // 初始化网段Ping网格
        if (document.getElementById('segment-ping-grid')) {
            initSegmentGrid();
        }
        
        // 初始化端口类型选择
        if (document.getElementById('scan-port-type')) {
            onPortTypeChange();
        }
        
        // 初始化会话数测试服务器选择
        const serverRadios = document.querySelectorAll('input[name="session-server"]');
        const customInput = document.getElementById('session-custom-input');
        if (serverRadios.length > 0 && customInput) {
            serverRadios.forEach(radio => {
                radio.addEventListener('change', function() {
                    if (this.value === 'custom') {
                        customInput.style.display = 'flex';
                    } else {
                        customInput.style.display = 'none';
                    }
                });
            });
        }
    });
    
    // 页面加载时更新授权状态
    updateLicenseStatus();
});

function initNavigation() {
    const navItems = document.querySelectorAll('.nav-item');
    navItems.forEach(item => {
        item.addEventListener('click', function(e) {
            e.preventDefault();
            const page = this.getAttribute('data-page');
            switchPage(page);
            
            navItems.forEach(ni => ni.classList.remove('active'));
            this.classList.add('active');
        });
    });
}

function switchPage(page) {
    document.querySelectorAll('.page').forEach(p => p.classList.remove('active'));
    const targetPage = document.getElementById(`page-${page}`);
    if (targetPage) {
        targetPage.classList.add('active');
        
        if (page === 'dashboard') {
            initDashboard();
        } else if (page === 'dhcp') {
            loadInterfaces();
        } else if (page === 'networktools') {
            // 延迟初始化，避免阻塞页面切换
            requestAnimationFrame(() => {
                if (document.getElementById('page-networktools')) {
                    switchToolsTab('dns');
                }
            });
        } else if (page === 'packet') {
            loadPacketInterfaces();
        } else if (page === 'loop') {
            loadLoopInterfaces();
        } else if (page === 'camera') {
            clearCameraResult();
        } else if (page === 'session') {
            clearSessionLog();
        } else if (page === 'localsettings') {
            switchSettingsTab('network');
        } else if (page === 'about') {
            switchAboutTab('support');
        } else if (page === 'ai') {
            checkAILicense().then(() => {
                switchAITab('chat');
                loadAIConfig();
            }).catch(() => {
                // 未授权，显示提示
                alert('AI功能需要授权才能使用，请先激活授权');
            });
        } else if (page === 'packet') {
            loadPacketInterfaces();
        }
    }
}

// 导出switchPage供HTML使用（其他函数在文件末尾统一导出）
window.switchPage = switchPage;

function initDashboard() {
    if (dashboardInterval) {
        clearInterval(dashboardInterval);
    }
    // 延迟首次更新，避免阻塞页面渲染
    requestAnimationFrame(() => {
        updateDashboard();
        dashboardInterval = setInterval(updateDashboard, 2000);
    });
}

function updateDashboard() {
    fetch('/api/dashboard/stats')
        .then(res => {
            if (!res.ok) {
                throw new Error(`HTTP error! status: ${res.status}`);
            }
            return res.json();
        })
        .then(data => {
            if (data.success) {
                const s = data.data;
                
                // CPU
                if (s.cpu) {
                    const cpuEl = document.getElementById('stat-cpu');
                    const cpuDescEl = document.getElementById('stat-cpu-desc');
                    if (cpuEl) cpuEl.textContent = `${s.cpu.percent.toFixed(1)}%`;
                    if (cpuDescEl) cpuDescEl.textContent = `${s.cpu.count || 0} 核心`;
                }
                
                // 内存
                if (s.memory) {
                    const memPercent = (s.memory.used / s.memory.total * 100).toFixed(1);
                    const memEl = document.getElementById('stat-memory');
                    const memDescEl = document.getElementById('stat-memory-desc');
                    if (memEl) memEl.textContent = `${memPercent}%`;
                    if (memDescEl) {
                        memDescEl.textContent = `${formatBytes(s.memory.used)} / ${formatBytes(s.memory.total)}`;
                    }
                }
                
                // 磁盘
                if (s.disk) {
                    const diskEl = document.getElementById('stat-disk');
                    const diskDescEl = document.getElementById('stat-disk-desc');
                    if (diskEl) diskEl.textContent = `${s.disk.percent.toFixed(1)}%`;
                    if (diskDescEl) {
                        diskDescEl.textContent = `${formatBytes(s.disk.used)} / ${formatBytes(s.disk.total)}`;
                    }
                }
                
                // 网络
                if (s.network) {
                    const recv = formatBytes(s.network.recv_speed || 0);
                    const send = formatBytes(s.network.send_speed || 0);
                    const networkEl = document.getElementById('stat-network');
                    if (networkEl) {
                        networkEl.textContent = `↓ ${recv} ↑ ${send}`;
                    }
                    
                    // 这些元素可能不存在（已删除的系统信息卡片中的元素）
                    const totalSentEl = document.getElementById('total-sent');
                    const totalRecvEl = document.getElementById('total-recv');
                    const packetsSentEl = document.getElementById('packets-sent');
                    const packetsRecvEl = document.getElementById('packets-recv');
                    
                    if (totalSentEl) totalSentEl.textContent = formatBytes(s.network.bytes_sent || 0);
                    if (totalRecvEl) totalRecvEl.textContent = formatBytes(s.network.bytes_recv || 0);
                    if (packetsSentEl) packetsSentEl.textContent = (s.network.packets_sent || 0).toLocaleString();
                    if (packetsRecvEl) packetsRecvEl.textContent = (s.network.packets_recv || 0).toLocaleString();
                }
                
                // 运行时间
                try {
                    if (s.uptime !== undefined && s.uptime !== null && s.uptime > 0) {
                        const uptimeEl = document.getElementById('stat-uptime');
                        if (uptimeEl) {
                            uptimeEl.textContent = formatUptime(s.uptime);
                        }
                    }
                } catch (e) {
                    console.error('更新运行时间失败:', e);
                }
                
                // 启动时间
                try {
                    if (s.boot_time) {
                        const bootTimeEl = document.getElementById('stat-boot-time');
                        if (bootTimeEl) {
                            bootTimeEl.textContent = `启动时间: ${s.boot_time}`;
                        }
                    }
                } catch (e) {
                    console.error('更新启动时间失败:', e);
                }
                
                // 进程数
                try {
                    if (s.processes !== undefined && s.processes !== null) {
                        const processesEl = document.getElementById('stat-processes');
                        if (processesEl) {
                            processesEl.textContent = s.processes.toLocaleString();
                        }
                    }
                } catch (e) {
                    console.error('更新进程数失败:', e);
                }
                
                // 系统负载（进程数卡片中的描述）
                try {
                    if (s.load_avg && typeof s.load_avg === 'object' && s.load_avg['1min'] !== undefined) {
                        const statLoadEl = document.getElementById('stat-load');
                        if (statLoadEl) {
                            statLoadEl.textContent = `负载: ${parseFloat(s.load_avg['1min']).toFixed(2)}`;
                        }
                    }
                } catch (e) {
                    console.error('更新系统负载失败:', e);
                }
                
                // 系统负载卡片
                try {
                    if (s.load_avg && typeof s.load_avg === 'object' && s.load_avg['1min'] !== undefined) {
                        const statLoadValueEl = document.getElementById('stat-load-value');
                        const statLoadDescEl = document.getElementById('stat-load-desc');
                        if (statLoadValueEl) {
                            statLoadValueEl.textContent = parseFloat(s.load_avg['1min']).toFixed(2);
                        }
                        if (statLoadDescEl && s.load_avg['5min'] !== undefined) {
                            statLoadDescEl.textContent = `5分钟: ${parseFloat(s.load_avg['5min']).toFixed(2)}`;
                        }
                    }
                } catch (e) {
                    console.error('更新系统负载卡片失败:', e);
                }
                
                // 网络连接数
                try {
                    if (s.connections && typeof s.connections === 'object' && s.connections.total !== undefined) {
                        const statConnEl = document.getElementById('stat-connections');
                        const statConnDescEl = document.getElementById('stat-connections-desc');
                        if (statConnEl) {
                            statConnEl.textContent = (s.connections.total || 0).toLocaleString();
                        }
                        if (statConnDescEl) {
                            statConnDescEl.textContent = `TCP: ${s.connections.tcp || 0} / UDP: ${s.connections.udp || 0}`;
                        }
                        
                        // 系统版本
                        const statVersionEl = document.getElementById('stat-version');
                        const statVersionDescEl = document.getElementById('stat-version-desc');
                        if (statVersionEl && s.system_version) {
                            statVersionEl.textContent = s.system_version.distro || 'Unknown';
                        }
                        if (statVersionDescEl && s.system_version) {
                            statVersionDescEl.textContent = s.system_version.kernel || 'Unknown';
                        }
                    }
                } catch (e) {
                    console.error('更新网络连接数失败:', e);
                }
                
                // 网络接口
                try {
                    if (s.interfaces && Array.isArray(s.interfaces)) {
                        const activeInterfaces = s.interfaces.filter(iface => iface && iface.isup && iface.ipv4 && iface.name !== 'lo');
                        const allInterfaces = s.interfaces.filter(iface => iface && iface.ipv4);
                        const interfacesCountEl = document.getElementById('interfaces-count');
                        const interfacesDescEl = document.getElementById('interfaces-desc');
                        
                        if (interfacesCountEl) {
                            interfacesCountEl.textContent = activeInterfaces.length > 0 ? activeInterfaces.length : allInterfaces.length;
                        }
                        if (interfacesDescEl) {
                            if (activeInterfaces.length > 0) {
                                interfacesDescEl.textContent = activeInterfaces.map(i => i.name).join(', ');
                            } else if (allInterfaces.length > 0) {
                                interfacesDescEl.textContent = allInterfaces.map(i => i.name).join(', ');
                            } else {
                                interfacesDescEl.textContent = '无接口';
                            }
                        }
                        
                        updateInterfacesList(s.interfaces);
                    }
                } catch (e) {
                    console.error('更新网络接口失败:', e);
                }
            } else {
                console.error('API返回失败:', data.error || '未知错误');
            }
        })
        .catch(err => {
            console.error('仪表板更新失败:', err);
        });
}

function formatUptime(seconds) {
    const days = Math.floor(seconds / 86400);
    const hours = Math.floor((seconds % 86400) / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    
    if (days > 0) {
        return `${days}天 ${hours}小时`;
    } else if (hours > 0) {
        return `${hours}小时 ${minutes}分钟`;
    } else {
        return `${minutes}分钟`;
    }
}

function updateInterfacesList(interfaces) {
    const container = document.getElementById('interfaces-list');
    if (!container) return;
    
    if (!interfaces || interfaces.length === 0) {
        container.innerHTML = '<div class="loading">未找到网络接口</div>';
        return;
    }
    
    // 过滤出有IP地址的接口（包括lo）
    const validInterfaces = interfaces.filter(iface => iface.ipv4);
    
    if (validInterfaces.length === 0) {
        container.innerHTML = '<div class="loading">未找到配置IP的网络接口</div>';
        return;
    }
    
    let html = '';
    validInterfaces.forEach(iface => {
        const statusClass = iface.isup ? 'up' : 'down';
        const statusText = iface.isup ? '启用' : '禁用';
        const speedText = iface.speed > 0 ? `${iface.speed} Mbps` : 'N/A';
        
        html += `
            <div class="interface-card ${iface.isup ? 'active' : ''}">
                <div class="interface-name">
                    <i class="fas fa-ethernet"></i>
                    ${iface.name}
                </div>
                <div class="interface-status ${statusClass}">${statusText}</div>
                <div class="interface-details">
                    <div><strong>IP地址:</strong> ${iface.ipv4 || 'N/A'}</div>
                    <div><strong>MAC地址:</strong> ${iface.mac || 'N/A'}</div>
                    <div><strong>速度:</strong> ${speedText}</div>
                </div>
            </div>
        `;
    });
    
    container.innerHTML = html;
}

function refreshInterfaces() {
    const container = document.getElementById('interfaces-list');
    container.innerHTML = '<div class="loading">刷新中...</div>';
    updateDashboard();
}


function handlePing() {
    const target = document.getElementById('ping-target').value.trim();
    const count = parseInt(document.getElementById('ping-count').value);
    const size = parseInt(document.getElementById('ping-size').value);
    
    if (!target) {
        alert('请输入目标地址');
        return;
    }
    
    const result = document.getElementById('ping-result');
    result.innerHTML = '<div style="text-align: center; padding: 40px; color: var(--text-light);">⏳ 正在测试，请稍候...</div>';
    
    fetch('/api/ping/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({target, count, packet_size: size})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showPingResult(data.data);
        } else {
            result.textContent = `错误: ${data.error || '测试失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function stopPing() {
    fetch('/api/ping/stop', {method: 'POST'});
}

function showPingResult(data) {
    const result = document.getElementById('ping-result');
    
    // 美化输出格式
    let html = `<div class="ping-result-header">`;
    html += `<div class="ping-target">🎯 目标: <strong>${data.target}</strong></div>`;
    html += `<div class="ping-params">📦 包大小: ${data.packet_size} 字节 | 🔢 测试次数: ${data.count}</div>`;
    html += `</div>`;
    
    html += `<div class="ping-stats">`;
    html += `<div class="stat-row">`;
    html += `<span class="stat-label">发送:</span> <span class="stat-value">${data.packets_sent}</span>`;
    html += `<span class="stat-label">接收:</span> <span class="stat-value">${data.packets_received}</span>`;
    html += `<span class="stat-label">丢失:</span> <span class="stat-value">${data.packets_sent - data.packets_received}</span>`;
    html += `<span class="stat-label">丢包率:</span> <span class="stat-value ${data.packet_loss > 0 ? 'warning' : 'success'}">${data.packet_loss.toFixed(1)}%</span>`;
    html += `</div>`;
    html += `</div>`;
    
    html += `<div class="ping-timing">`;
    html += `<div class="timing-row">`;
    html += `<span class="timing-label">最小延迟:</span> <span class="timing-value">${data.min_time.toFixed(2)} ms</span>`;
    html += `<span class="timing-label">平均延迟:</span> <span class="timing-value highlight">${data.avg_time.toFixed(2)} ms</span>`;
    html += `<span class="timing-label">最大延迟:</span> <span class="timing-value">${data.max_time.toFixed(2)} ms</span>`;
    html += `</div>`;
    html += `</div>`;
    
    if (data.details && data.details.length > 0) {
        html += `<div class="ping-details">`;
        html += `<div class="details-title">📊 详细结果:</div>`;
        html += `<div class="details-grid">`;
        data.details.forEach(d => {
            const statusClass = d.status === 'success' ? 'success' : 'failed';
            const statusIcon = d.status === 'success' ? '✅' : '❌';
            html += `<div class="detail-item ${statusClass}">`;
            html += `<span class="detail-seq">#${d.seq}</span>`;
            html += `<span class="detail-time">${d.time.toFixed(2)} ms</span>`;
            html += `<span class="detail-status">${statusIcon}</span>`;
            html += `</div>`;
        });
        html += `</div>`;
        html += `</div>`;
    }
    
    result.innerHTML = html;
}

// Ping选项卡切换
function switchPingTab(tab) {
    document.querySelectorAll('.ping-tab').forEach(t => t.classList.remove('active'));
    document.querySelectorAll('.ping-tab-content').forEach(c => c.classList.remove('active'));
    
    document.querySelector(`.ping-tab[data-tab="${tab}"]`).classList.add('active');
    document.getElementById(`ping-${tab}`).classList.add('active');
    
    // 如果是网段ping，初始化网格
    if (tab === 'segment') {
        initSegmentGrid();
    }
}

// 持续Ping
let continuousInterval = null;

function handleContinuousPing() {
    const target = document.getElementById('continuous-ping-target').value.trim();
    const size = parseInt(document.getElementById('continuous-ping-size').value);
    const interval = parseFloat(document.getElementById('continuous-ping-interval').value);
    
    if (!target) {
        alert('请输入目标地址');
        return;
    }
    
    const result = document.getElementById('continuous-ping-result');
    result.textContent = `开始持续Ping ${target}...\n`;
    
    fetch('/api/ping/continuous/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({target, packet_size: size, interval})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            // 开始轮询状态
            continuousInterval = setInterval(updateContinuousStatus, interval * 1000);
        } else {
            result.textContent += `错误: ${data.error}\n`;
        }
    })
    .catch(err => {
        result.textContent += `请求失败: ${err.message}\n`;
    });
}

function updateContinuousStatus() {
    fetch('/api/ping/continuous/status')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const stats = data.data.stats;
                const results = data.data.results;
                
                document.getElementById('continuous-stats').textContent = 
                    `已发送: ${stats.sent} | 成功: ${stats.success} | 失败: ${stats.failed} | 成功率: ${stats.sent > 0 ? ((stats.success / stats.sent) * 100).toFixed(1) : 0}%`;
                
                const result = document.getElementById('continuous-ping-result');
                if (results.length > 0) {
                    const last = results[results.length - 1];
                    const time = new Date().toLocaleTimeString();
                    const status = last.status === 'success' ? '✅' : '❌';
                    result.textContent += `[${time}] ${status} seq=${last.seq} time=${last.time}ms\n`;
                    result.scrollTop = result.scrollHeight;
                }
            }
        })
        .catch(err => console.error('获取持续Ping状态失败:', err));
}

function stopContinuousPing() {
    if (continuousInterval) {
        clearInterval(continuousInterval);
        continuousInterval = null;
    }
    fetch('/api/ping/stop', {method: 'POST'});
}

function clearContinuousResult() {
    document.getElementById('continuous-ping-result').textContent = '';
}

function exportContinuousResult() {
    const result = document.getElementById('continuous-ping-result').textContent;
    const stats = document.getElementById('continuous-stats').textContent;
    const target = document.getElementById('continuous-ping-target').value;
    const size = document.getElementById('continuous-ping-size').value;
    const interval = document.getElementById('continuous-ping-interval').value;
    
    if (!result || result.trim() === '等待测试...') {
        alert('没有可导出的结果');
        return;
    }
    
    const timestamp = new Date().toLocaleString('zh-CN');
    let content = '='.repeat(60) + '\n';
    content += '持续Ping测试结果导出\n';
    content += '='.repeat(60) + '\n';
    content += `导出时间: ${timestamp}\n`;
    content += `目标地址: ${target}\n`;
    content += `包大小: ${size} 字节\n`;
    content += `间隔时间: ${interval} 秒\n`;
    content += `统计信息: ${stats}\n`;
    content += '='.repeat(60) + '\n\n';
    content += result;
    
    // 创建下载链接
    const blob = new Blob([content], { type: 'text/plain;charset=utf-8' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `持续Ping结果_${target}_${new Date().toISOString().slice(0,10)}.txt`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
}

// 批量Ping
let batchInterval = null;

function handleBatchPing() {
    const hostsText = document.getElementById('batch-ping-hosts').value.trim();
    const hosts = hostsText.split('\n').filter(h => h.trim() && !h.trim().startsWith('#')).map(h => h.trim());
    const size = parseInt(document.getElementById('batch-ping-size').value);
    const count = parseInt(document.getElementById('batch-ping-count').value);
    
    if (hosts.length === 0) {
        alert('请输入主机列表');
        return;
    }
    
    // 重置结果
    batchPingFullResults = [];
    const result = document.getElementById('batch-ping-result');
    result.textContent = `🚀 开始批量Ping ${hosts.length} 个主机...\n${'='.repeat(60)}\n\n`;
    
    fetch('/api/ping/batch/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({hosts, packet_size: size, count})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            batchInterval = setInterval(updateBatchStatus, 1000);
        } else {
            result.textContent += `错误: ${data.error}\n`;
        }
    })
    .catch(err => {
        result.textContent += `请求失败: ${err.message}\n`;
    });
}

// 存储批量ping的完整结果
let batchPingFullResults = [];

function updateBatchStatus() {
    fetch('/api/ping/batch/status')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const stats = data.data.stats;
                const results = data.data.results;
                
                // 更新统计信息
                document.getElementById('batch-stats').textContent = 
                    `总主机: ${stats.total} | 已测试: ${stats.tested} | 成功: ${stats.success} | 失败: ${stats.failed} | 成功率: ${stats.tested > 0 ? ((stats.success / stats.tested) * 100).toFixed(1) : 0}%`;
                
                // 保存完整结果用于导出
                batchPingFullResults = results;
                
                const result = document.getElementById('batch-ping-result');
                let content = `🚀 开始批量Ping ${stats.total} 个主机\n`;
                content += `${'='.repeat(60)}\n\n`;
                
                // 按索引排序显示
                const sortedResults = [...results].sort((a, b) => (a.index || 0) - (b.index || 0));
                
                sortedResults.forEach(r => {
                    if (r.host) {
                        const status = r.status === 'success' ? '✅' : '❌';
                        content += `[${r.index}/${r.total}] ${status} ${r.host}`;
                        
                        if (r.result) {
                            if (r.result.avg_time) {
                                content += ` - 延迟: ${r.result.avg_time.toFixed(2)}ms`;
                            }
                            if (r.result.packet_loss !== undefined) {
                                content += ` - 丢包率: ${r.result.packet_loss.toFixed(1)}%`;
                            }
                            if (r.result.packets_received !== undefined) {
                                content += ` (${r.result.packets_received}/${r.result.packets_sent})`;
                            }
                        }
                        content += '\n';
                    }
                });
                
                // 如果测试完成，添加总结
                if (stats.tested >= stats.total) {
                    content += `\n${'='.repeat(60)}\n`;
                    content += `✅ 批量Ping完成\n`;
                    content += `总计: ${stats.total} | 成功: ${stats.success} | 失败: ${stats.failed}\n`;
                    content += `成功率: ${((stats.success / stats.total) * 100).toFixed(1)}%\n`;
                    
                    if (batchInterval) {
                        clearInterval(batchInterval);
                        batchInterval = null;
                    }
                } else {
                    content += `\n⏳ 测试中... (${stats.tested}/${stats.total})\n`;
                }
                
                result.textContent = content;
                result.scrollTop = result.scrollHeight;
            }
        })
        .catch(err => console.error('获取批量Ping状态失败:', err));
}

function stopBatchPing() {
    if (batchInterval) {
        clearInterval(batchInterval);
        batchInterval = null;
    }
    fetch('/api/ping/stop', {method: 'POST'});
}

function clearBatchResult() {
    document.getElementById('batch-ping-result').textContent = '';
    batchPingFullResults = [];
}

function exportBatchResult() {
    const result = document.getElementById('batch-ping-result').textContent;
    const stats = document.getElementById('batch-stats').textContent;
    const hostsText = document.getElementById('batch-ping-hosts').value.trim();
    const size = document.getElementById('batch-ping-size').value;
    const count = document.getElementById('batch-ping-count').value;
    
    if (!result || result.trim() === '等待测试...' || result.trim() === '') {
        alert('没有可导出的结果');
        return;
    }
    
    const timestamp = new Date().toLocaleString('zh-CN');
    let content = '='.repeat(60) + '\n';
    content += '批量Ping测试结果导出\n';
    content += '='.repeat(60) + '\n';
    content += `导出时间: ${timestamp}\n`;
    content += `主机列表:\n${hostsText.split('\n').map((h, i) => `  ${i+1}. ${h}`).join('\n')}\n`;
    content += `包大小: ${size} 字节\n`;
    content += `每个主机次数: ${count}\n`;
    content += `统计信息: ${stats}\n`;
    content += '='.repeat(60) + '\n\n';
    content += result;
    
    // 如果有详细结果，添加详细信息
    if (batchPingFullResults.length > 0) {
        content += '\n\n' + '='.repeat(60) + '\n';
        content += '详细结果:\n';
        content += '='.repeat(60) + '\n';
        
        batchPingFullResults.forEach(r => {
            if (r.host && r.result) {
                content += `\n主机: ${r.host}\n`;
                content += `  状态: ${r.status === 'success' ? '成功' : '失败'}\n`;
                if (r.result.packets_sent) {
                    content += `  发送: ${r.result.packets_sent} | 接收: ${r.result.packets_received || 0}\n`;
                }
                if (r.result.packet_loss !== undefined) {
                    content += `  丢包率: ${r.result.packet_loss.toFixed(1)}%\n`;
                }
                if (r.result.min_time !== undefined) {
                    content += `  延迟: 最小 ${r.result.min_time.toFixed(2)}ms | 平均 ${r.result.avg_time.toFixed(2)}ms | 最大 ${r.result.max_time.toFixed(2)}ms\n`;
                }
            }
        });
    }
    
    // 创建下载链接
    const blob = new Blob([content], { type: 'text/plain;charset=utf-8' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `批量Ping结果_${new Date().toISOString().slice(0,10)}_${new Date().toTimeString().slice(0,8).replace(/:/g,'')}.txt`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
}

// 网段Ping
let segmentInterval = null;
let segmentCells = {};

function initSegmentGrid() {
    const grid = document.getElementById('segment-ping-grid');
    if (!grid || grid.children.length > 0) return; // 已经初始化
    
    grid.innerHTML = '';
    segmentCells = {};
    
    for (let i = 1; i <= 255; i++) {
        const cell = document.createElement('div');
        cell.className = 'segment-cell';
        cell.dataset.ipNum = i;
        cell.innerHTML = `
            <div class="segment-cell-number">${i}</div>
            <div class="segment-cell-status"></div>
        `;
        grid.appendChild(cell);
        segmentCells[i] = cell;
    }
}

function handleSegmentPing() {
    const network = document.getElementById('segment-ping-network').value.trim();
    const size = parseInt(document.getElementById('segment-ping-size').value);
    const timeout = parseFloat(document.getElementById('segment-ping-timeout').value);
    const threads = parseInt(document.getElementById('segment-ping-threads').value);
    
    if (!network) {
        alert('请输入网段');
        return;
    }
    
    // 初始化网格
    initSegmentGrid();
    
    // 重置所有格子
    for (let i = 1; i <= 255; i++) {
        const cell = segmentCells[i];
        if (cell) {
            cell.className = 'segment-cell';
            cell.querySelector('.segment-cell-status').textContent = '';
        }
    }
    
    fetch('/api/ping/segment/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({network, packet_size: size, timeout, threads})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            segmentInterval = setInterval(updateSegmentStatus, 500);
        } else {
            alert(`错误: ${data.error}`);
        }
    })
    .catch(err => {
        alert(`请求失败: ${err.message}`);
    });
}

function updateSegmentStatus() {
    fetch('/api/ping/segment/status')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const stats = data.data.stats;
                const results = data.data.results;
                
                // 更新统计信息
                const statsEl = document.getElementById('segment-stats');
                if (statsEl) {
                    const total = stats.total || 254; // 默认254（/24网段）
                    const onlineRate = stats.tested > 0 ? ((stats.online / stats.tested) * 100).toFixed(1) : 0;
                    statsEl.textContent = 
                        `总计: ${total} | 已扫: ${stats.tested} | 在线: ${stats.online} | 离线: ${stats.offline} | 在线率: ${onlineRate}%`;
                }
                
                // 更新格子状态（只更新有结果的）
                Object.keys(results).forEach(ipNumStr => {
                    const result = results[ipNumStr];
                    const ipNum = parseInt(ipNumStr);
                    const cell = segmentCells[ipNum];
                    if (cell && result) {
                        // 移除所有状态类，添加新状态
                        cell.className = 'segment-cell';
                        if (result.status === 'online') {
                            cell.classList.add('online');
                            const statusEl = cell.querySelector('.segment-cell-status');
                            if (statusEl) statusEl.textContent = '🟢';
                        } else if (result.status === 'offline') {
                            cell.classList.add('offline');
                            const statusEl = cell.querySelector('.segment-cell-status');
                            if (statusEl) statusEl.textContent = '🔴';
                        } else if (result.status === 'testing') {
                            cell.classList.add('testing');
                            const statusEl = cell.querySelector('.segment-cell-status');
                            if (statusEl) statusEl.textContent = '🟡';
                        }
                    }
                });
                
                // 如果扫描完成，停止轮询
                const total = stats.total || 254;
                if (stats.tested >= total) {
                    if (segmentInterval) {
                        clearInterval(segmentInterval);
                        segmentInterval = null;
                    }
                }
            }
        })
        .catch(err => console.error('获取网段Ping状态失败:', err));
}

function stopSegmentPing() {
    if (segmentInterval) {
        clearInterval(segmentInterval);
        segmentInterval = null;
    }
    fetch('/api/ping/stop', {method: 'POST'});
}

function handleTracert() {
    const target = document.getElementById('tracert-target').value.trim();
    const hops = parseInt(document.getElementById('tracert-hops').value);
    
    if (!target) {
        alert('请输入目标地址');
        return;
    }
    
    const result = document.getElementById('tracert-result');
    result.textContent = '正在追踪路由，请稍候...';
    
    fetch('/api/tracert/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({target, max_hops: hops})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showTracertResult(data.data);
        } else {
            result.textContent = `错误: ${data.error || '追踪失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function stopTracert() {
    fetch('/api/tracert/stop', {method: 'POST'});
}

function showTracertResult(data) {
    const result = document.getElementById('tracert-result');
    const target = data.target || '未知';
    const maxHops = data.max_hops || 30;
    let text = `目标: ${target}\n最大跳数: ${maxHops}\n\n`;
    
    if (data.hops && data.hops.length > 0) {
        data.hops.forEach(hop => {
            text += `${hop.hop}.  `;
            if (hop.ip) {
                // 显示主机名（如果有）
                if (hop.hostname) {
                    text += `${hop.hostname} (${hop.ip})`;
                } else {
                    text += hop.ip;
                }
                
                // 显示地理位置（如果有）
                if (hop.location) {
                    text += `  [${hop.location}]`;
                }
                
                // 显示时间
                text += `  `;
                if (hop.avg_time !== null && hop.avg_time !== undefined) {
                    text += `${hop.avg_time.toFixed(2)}ms`;
                } else if (hop.times && hop.times.length > 0) {
                    // 如果有多个时间值，显示第一个或平均
                    const avg = hop.times.reduce((a, b) => a + b, 0) / hop.times.length;
                    text += `${avg.toFixed(2)}ms`;
                }
            } else {
                text += `*  超时`;
            }
            text += `\n`;
        });
    } else {
        text += '未找到路由路径';
    }
    
    result.textContent = text;
}

function onPortTypeChange() {
    const portType = document.getElementById('scan-port-type').value;
    const portsInput = document.getElementById('scan-ports');
    
    if (portType === 'common') {
        // 常规端口：常用端口列表
        portsInput.value = '21,22,23,25,53,80,110,143,443,993,995,135,139,445,3389,3306,5432,1433,6379,8080,8443';
        portsInput.disabled = true;
    } else if (portType === 'all') {
        // 所有端口：1-65535
        portsInput.value = '1-65535';
        portsInput.disabled = true;
    } else {
        // 自定义：允许用户输入
        portsInput.value = '';
        portsInput.disabled = false;
        portsInput.placeholder = '80,443 或 1-1000';
    }
}

function handlePortScan() {
    const target = document.getElementById('scan-target').value.trim();
    const portType = document.getElementById('scan-port-type').value;
    let ports = document.getElementById('scan-ports').value.trim();
    const threads = parseInt(document.getElementById('scan-threads').value);
    
    if (!target) {
        alert('请输入目标地址');
        return;
    }
    
    // 根据端口类型设置端口范围
    if (portType === 'common') {
        ports = '21,22,23,25,53,80,110,143,443,993,995,135,139,445,3389,3306,5432,1433,6379,8080,8443';
    } else if (portType === 'all') {
        ports = '1-65535';
    }
    
    if (!ports) {
        alert('请输入端口范围');
        return;
    }
    
    const checkVulnerability = document.getElementById('scan-vulnerability').checked;
    const result = document.getElementById('scan-result');
    result.textContent = '正在扫描端口，请稍候...';
    
    fetch('/api/portscan/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({target, ports, threads, timeout: 1.0, check_vulnerability: checkVulnerability})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showScanResult(data.data);
        } else {
            result.textContent = `错误: ${data.error || '扫描失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function stopPortScan() {
    fetch('/api/portscan/stop', {method: 'POST'});
}

function showScanResult(data) {
    const result = document.getElementById('scan-result');
    let html = `<div style="margin-bottom: 10px;"><strong>目标:</strong> ${data.target} (${data.target_ip})</div>`;
    html += `<div style="margin-bottom: 10px;"><strong>扫描端口数:</strong> ${data.total_ports}  <strong>开放端口:</strong> ${data.open_count}</div>`;
    
    if (data.open_ports && data.open_ports.length > 0) {
        html += `<div style="margin-top: 15px;"><strong>开放端口列表:</strong></div>`;
        html += `<table style="width: 100%; margin-top: 10px; border-collapse: collapse;">`;
        html += `<tr style="background: #f5f7fa;"><th style="padding: 8px; text-align: left;">端口</th><th style="padding: 8px; text-align: left;">服务/版本</th><th style="padding: 8px; text-align: left;">状态</th><th style="padding: 8px; text-align: left;">漏洞 (CVE/CWE/OWASP)</th><th style="padding: 8px; text-align: left;">解决方案</th></tr>`;
        data.open_ports.forEach(p => {
            const serviceInfo = p.version || p.service || 'Unknown';
            let vulnCell = '<span style="color: #67c23a;">无</span>';
            let solutionCell = '<span style="color: #909399;">-</span>';
            
            // 收集所有漏洞和安全问题
            let allIssues = [];
            let allSolutions = [];
            
            // 处理CVE漏洞信息（支持新格式和旧格式）
            if (p.vulnerability_details && p.vulnerability_details.length > 0) {
                // 新格式：包含id和solution的对象数组
                p.vulnerability_details.forEach(v => {
                    allIssues.push({type: 'CVE', id: v.id || v, name: v.id || v});
                    if (v.solution) allSolutions.push(v.solution);
                });
            } else if (p.vulnerabilities && p.vulnerabilities.length > 0) {
                // 旧格式：字符串数组
                p.vulnerabilities.forEach(v => {
                    const id = typeof v === 'string' ? v : v.id || v;
                    allIssues.push({type: 'CVE', id: id, name: id});
                });
            }
            
            // 处理其他安全问题（CWE、OWASP等）
            if (p.security_issues && p.security_issues.length > 0) {
                p.security_issues.forEach(issue => {
                    allIssues.push({
                        type: issue.type || 'ISSUE',
                        id: issue.id || issue.name,
                        name: issue.name || issue.id
                    });
                    if (issue.solution) allSolutions.push(issue.solution);
                });
            }
            
            // 构建显示内容
            if (allIssues.length > 0) {
                // 按类型分组显示
                const cveIssues = allIssues.filter(i => i.type === 'CVE');
                const otherIssues = allIssues.filter(i => i.type !== 'CVE');
                
                let issueText = [];
                if (cveIssues.length > 0) {
                    const cveIds = cveIssues.map(i => i.id).join(', ');
                    issueText.push(`<span style="color: #f56c6c;">CVE: ${cveIds}</span>`);
                }
                if (otherIssues.length > 0) {
                    const otherIds = otherIssues.map(i => `${i.type}: ${i.id}`).join(', ');
                    issueText.push(`<span style="color: #e6a23c;">${otherIds}</span>`);
                }
                
                vulnCell = `<span style="color: #f56c6c; font-weight: bold;">⚠️ ${allIssues.length}个</span><br><small style="color: #606266; white-space: nowrap; line-height: 1.6;">${issueText.join('<br>')}</small>`;
                
                // 显示解决方案（去重，最多显示3个）
                const uniqueSolutions = [...new Set(allSolutions)].slice(0, 3);
                if (uniqueSolutions.length > 0) {
                    solutionCell = `<small style="color: #409eff; line-height: 1.6;">${uniqueSolutions.join('<br>')}</small>`;
                }
            }
            
            html += `<tr>
                <td style="padding: 8px; border-bottom: 1px solid #eee;">${p.port}</td>
                <td style="padding: 8px; border-bottom: 1px solid #eee;">${serviceInfo}</td>
                <td style="padding: 8px; border-bottom: 1px solid #eee; color: #67c23a;">开放</td>
                <td style="padding: 8px; border-bottom: 1px solid #eee;">${vulnCell}</td>
                <td style="padding: 8px; border-bottom: 1px solid #eee; max-width: 300px;">${solutionCell}</td>
            </tr>`;
        });
        html += `</table>`;
    } else {
        html += `<div style="margin-top: 15px; color: #909399;">未发现开放端口</div>`;
    }
    
    result.innerHTML = html;
}

function handleHostDiscovery() {
    const network = document.getElementById('discovery-network').value.trim();
    const type = document.getElementById('discovery-type').value;
    const threads = parseInt(document.getElementById('discovery-threads').value);
    
    if (!network) {
        alert('请输入网段地址');
        return;
    }
    
    const result = document.getElementById('discovery-result');
    result.textContent = '正在扫描主机，请稍候...';
    
    fetch('/api/hostdiscovery/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({network, scan_type: type, threads})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showDiscoveryResult(data.data);
        } else {
            result.textContent = `错误: ${data.error || '扫描失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function stopHostDiscovery() {
    fetch('/api/hostdiscovery/stop', {method: 'POST'});
}

function showDiscoveryResult(data) {
    const result = document.getElementById('discovery-result');
    let html = `<div style="margin-bottom: 10px;"><strong>网段:</strong> ${data.network}</div>`;
    html += `<div style="margin-bottom: 10px;"><strong>总主机数:</strong> ${data.total_hosts}  <strong>存活主机:</strong> ${data.alive_count}</div>`;
    
    if (data.alive_hosts && data.alive_hosts.length > 0) {
        html += `<div style="margin-top: 15px;"><strong>存活主机列表:</strong></div>`;
        html += `<table style="width: 100%; margin-top: 10px; border-collapse: collapse;">`;
        html += `<tr style="background: #f5f7fa;"><th style="padding: 8px; text-align: left;">IP地址</th><th style="padding: 8px; text-align: left;">MAC地址</th><th style="padding: 8px; text-align: left;">方法</th></tr>`;
        data.alive_hosts.forEach(h => {
            html += `<tr><td style="padding: 8px; border-bottom: 1px solid #eee;">${h.ip}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${h.mac || 'N/A'}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${h.method || 'N/A'}</td></tr>`;
        });
        html += `</table>`;
    } else {
        html += `<div style="margin-top: 15px; color: #909399;">未发现存活主机</div>`;
    }
    
    result.innerHTML = html;
}

function switchSubnetTab(tab, event) {
    // 隐藏所有标签页内容
    document.querySelectorAll('.subnet-tab-content').forEach(el => {
        el.style.display = 'none';
    });
    
    // 移除所有按钮的active类
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // 显示选中的标签页
    document.getElementById(`subnet-${tab}`).style.display = 'block';
    
    // 激活对应的按钮
    if (event && event.target) {
        event.target.classList.add('active');
    } else {
        // 如果没有事件对象，通过按钮文本查找
        document.querySelectorAll('.tab-btn').forEach(btn => {
            if (btn.textContent.trim() === tab.toUpperCase() || 
                (tab === 'ipv4' && btn.textContent.includes('IPv4')) ||
                (tab === 'ipv6' && btn.textContent.includes('IPv6')) ||
                (tab === 'vlsm' && btn.textContent.includes('VLSM')) ||
                (tab === 'divide' && btn.textContent.includes('子网划分')) ||
                (tab === 'summary' && btn.textContent.includes('路由汇总'))) {
                btn.classList.add('active');
            }
        });
    }
}

function handleSubnetIPv4() {
    const network = document.getElementById('subnet-ipv4-network').value.trim();
    
    if (!network) {
        alert('请输入网络地址');
        return;
    }
    
    const result = document.getElementById('subnet-ipv4-result');
    result.textContent = '正在计算...';
    
    fetch('/api/subnet/calculate_ipv4', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({network})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showSubnetIPv4Result(data.data);
        } else {
            result.textContent = `错误: ${data.error || '计算失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function showSubnetIPv4Result(data) {
    const result = document.getElementById('subnet-ipv4-result');
    let html = `<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
        <tr style="background: #f5f7fa;">
            <th style="padding: 8px; border: 1px solid #ddd; text-align: left; width: 120px;">项目</th>
            <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">值</th>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">网络地址</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.network_address}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">广播地址</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.broadcast_address}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">子网掩码</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.netmask}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">主机掩码</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.hostmask}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">前缀长度</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">/${data.prefixlen}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">总地址数</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.total_addresses}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">可用主机</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.usable_hosts}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">网络类别</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.network_class || 'N/A'}</td>
        </tr>
    </table>`;
    result.innerHTML = html;
}

function handleSubnetIPv6() {
    const network = document.getElementById('subnet-ipv6-network').value.trim();
    
    if (!network) {
        alert('请输入网络地址');
        return;
    }
    
    const result = document.getElementById('subnet-ipv6-result');
    result.textContent = '正在计算...';
    
    fetch('/api/subnet/calculate_ipv6', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({network})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showSubnetIPv6Result(data.data);
        } else {
            result.textContent = `错误: ${data.error || '计算失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function showSubnetIPv6Result(data) {
    const result = document.getElementById('subnet-ipv6-result');
    let html = `<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
        <tr style="background: #f5f7fa;">
            <th style="padding: 8px; border: 1px solid #ddd; text-align: left; width: 120px;">项目</th>
            <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">值</th>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">网络地址</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.network_address}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">子网前缀</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">/${data.prefixlen}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">主机数量</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.total_addresses}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">可用主机</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.usable_hosts}</td>
        </tr>
        <tr>
            <td style="padding: 6px 8px; border: 1px solid #ddd; font-weight: bold;">地址类型</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd; color: #409eff;">${data.address_type || 'N/A'}</td>
        </tr>
    </table>`;
    result.innerHTML = html;
}

function handleVLSM() {
    const network = document.getElementById('vlsm-network').value.trim();
    const requirements = document.getElementById('vlsm-requirements').value.trim();
    
    if (!network) {
        alert('请输入主网络地址');
        return;
    }
    
    if (!requirements) {
        alert('请输入子网需求');
        return;
    }
    
    const result = document.getElementById('vlsm-result');
    result.textContent = '正在计算...';
    
    fetch('/api/subnet/calculate_vlsm', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({network, requirements})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showVLSMResult(data.data);
        } else {
            result.textContent = `错误: ${data.error || '计算失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function showVLSMResult(data) {
    const result = document.getElementById('vlsm-result');
    let html = `<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
        <tr style="background: #f5f7fa;">
            <th style="padding: 8px; border: 1px solid #ddd;">子网名称</th>
            <th style="padding: 8px; border: 1px solid #ddd;">需求主机数</th>
            <th style="padding: 8px; border: 1px solid #ddd;">网络地址</th>
            <th style="padding: 8px; border: 1px solid #ddd;">子网掩码</th>
            <th style="padding: 8px; border: 1px solid #ddd;">首个可用IP</th>
            <th style="padding: 8px; border: 1px solid #ddd;">最后可用IP</th>
            <th style="padding: 8px; border: 1px solid #ddd;">广播地址</th>
        </tr>`;
    
    data.subnets.forEach(subnet => {
        html += `<tr>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.name}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.hosts_needed}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.network}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.mask}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.first_ip}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.last_ip}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.broadcast}</td>
        </tr>`;
    });
    
    html += `</table>`;
    html += `<div style="margin-top: 10px; color: #909399;">已使用: ${data.total_used} / ${data.total_available} 地址</div>`;
    result.innerHTML = html;
}

function clearVLSM() {
    document.getElementById('vlsm-requirements').value = '';
    document.getElementById('vlsm-result').textContent = '等待计算...';
}

function handleDivide() {
    const network = document.getElementById('divide-network').value.trim();
    const numSubnets = parseInt(document.getElementById('divide-count').value);
    
    if (!network) {
        alert('请输入网络地址');
        return;
    }
    
    if (!numSubnets || numSubnets <= 0) {
        alert('请输入有效的子网数量');
        return;
    }
    
    const result = document.getElementById('divide-result');
    result.textContent = '正在计算...';
    
    fetch('/api/subnet/calculate_divide', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({network, num_subnets: numSubnets})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showDivideResult(data.data);
        } else {
            result.textContent = `错误: ${data.error || '计算失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function showDivideResult(data) {
    const result = document.getElementById('divide-result');
    let html = `<div style="margin-bottom: 10px;"><strong>共划分 ${data.total_subnets} 个子网:</strong></div>`;
    html += `<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
        <tr style="background: #f5f7fa;">
            <th style="padding: 8px; border: 1px solid #ddd;">网络地址</th>
            <th style="padding: 8px; border: 1px solid #ddd;">子网掩码</th>
            <th style="padding: 8px; border: 1px solid #ddd;">前缀</th>
            <th style="padding: 8px; border: 1px solid #ddd;">广播地址</th>
            <th style="padding: 8px; border: 1px solid #ddd;">首个可用IP</th>
            <th style="padding: 8px; border: 1px solid #ddd;">最后可用IP</th>
            <th style="padding: 8px; border: 1px solid #ddd;">可用主机数</th>
        </tr>`;
    
    data.subnets.forEach(subnet => {
        html += `<tr>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.network}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.mask}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.prefix}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.broadcast}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.first_ip}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.last_ip}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${subnet.hosts}</td>
        </tr>`;
    });
    
    html += `</table>`;
    result.innerHTML = html;
}

function handleSummary() {
    const networksText = document.getElementById('summary-networks').value.trim();
    
    if (!networksText) {
        alert('请输入网络地址列表');
        return;
    }
    
    const networks = networksText.split('\n').filter(line => line.trim());
    
    if (networks.length === 0) {
        alert('请输入至少一个网络地址');
        return;
    }
    
    const result = document.getElementById('summary-result');
    result.textContent = '正在计算...';
    
    fetch('/api/subnet/calculate_summary', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({networks})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showSummaryResult(data.data);
        } else {
            result.textContent = `错误: ${data.error || '计算失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function showSummaryResult(data) {
    const result = document.getElementById('summary-result');
    let html = `<div style="margin-bottom: 10px;">
        <strong>原始网络数:</strong> ${data.original_count} | 
        <strong>汇总后:</strong> ${data.summary_count} 个网络
    </div>`;
    html += `<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
        <tr style="background: #f5f7fa;">
            <th style="padding: 8px; border: 1px solid #ddd;">网络地址</th>
            <th style="padding: 8px; border: 1px solid #ddd;">子网掩码</th>
            <th style="padding: 8px; border: 1px solid #ddd;">前缀</th>
            <th style="padding: 8px; border: 1px solid #ddd;">广播地址</th>
        </tr>`;
    
    data.summary.forEach(net => {
        html += `<tr>
            <td style="padding: 8px; border: 1px solid #ddd;">${net.network}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${net.mask}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${net.prefix}</td>
            <td style="padding: 8px; border: 1px solid #ddd;">${net.broadcast}</td>
        </tr>`;
    });
    
    html += `</table>`;
    result.innerHTML = html;
}

// 向后兼容
function handleSubnetCalc() {
    handleSubnetIPv4();
}

function getLocalIP() {
    const result = document.getElementById('ipinfo-result');
    result.innerHTML = '<div style="color: #909399;">正在获取本机IP信息...</div>';
    
    fetch('/api/ipinfo/local')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                showIPInfo(data.data, '本机');
            } else {
                result.innerHTML = `<div style="color: #f56c6c;">错误: ${data.error || '获取失败'}</div>`;
            }
        })
        .catch(err => {
            result.innerHTML = `<div style="color: #f56c6c;">请求失败: ${err.message}</div>`;
        });
}

function getPublicIP() {
    const result = document.getElementById('ipinfo-result');
    result.innerHTML = '<div style="color: #909399;">正在获取公网IP信息...</div>';
    
    fetch('/api/ipinfo/public')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                showIPInfo(data.data, '公网');
            } else {
                result.innerHTML = `<div style="color: #f56c6c;">错误: ${data.error || '获取失败'}</div>`;
            }
        })
        .catch(err => {
            result.innerHTML = `<div style="color: #f56c6c;">请求失败: ${err.message}</div>`;
        });
}

function queryIP() {
    const ip = document.getElementById('query-ip').value.trim();
    
    if (!ip) {
        alert('请输入IP地址');
        return;
    }
    
    const result = document.getElementById('ipinfo-result');
    result.innerHTML = '<div style="color: #909399;">正在查询...</div>';
    
    fetch('/api/ipinfo/query', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({ip})
    })
    .then(res => res.json())
        .then(data => {
            if (data.success) {
                showIPInfo(data.data, '查询');
            } else {
                result.innerHTML = `<div style="color: #f56c6c;">错误: ${data.error || '查询失败'}</div>`;
            }
        })
        .catch(err => {
            result.innerHTML = `<div style="color: #f56c6c;">请求失败: ${err.message}</div>`;
        });
}

function switchIPInfoTab(tab, event) {
    if (event) event.preventDefault();
    
    // 隐藏所有内容
    document.getElementById('ipinfo-content-info').style.display = 'none';
    document.getElementById('ipinfo-content-conflict').style.display = 'none';
    
    // 重置所有按钮样式
    document.getElementById('ipinfo-tab-info').classList.remove('active');
    document.getElementById('ipinfo-tab-conflict').classList.remove('active');
    
    // 显示选中的标签页
    if (tab === 'info') {
        document.getElementById('ipinfo-content-info').style.display = 'block';
        document.getElementById('ipinfo-tab-info').classList.add('active');
    } else if (tab === 'conflict') {
        document.getElementById('ipinfo-content-conflict').style.display = 'block';
        document.getElementById('ipinfo-tab-conflict').classList.add('active');
    }
}

function refreshAllIP() {
    const result = document.getElementById('ipinfo-result');
    result.innerHTML = '<div style="color: #909399;">正在刷新所有IP信息...</div>';
    
    getLocalIP();
    setTimeout(() => {
        getPublicIP();
    }, 100);
}

let conflictDetectionRunning = false;

function startConflictDetection() {
    const ipRange = document.getElementById('conflict-range').value.trim();
    
    if (!ipRange) {
        alert('请输入IP范围');
        return;
    }
    
    conflictDetectionRunning = true;
    const statusEl = document.getElementById('conflict-status');
    statusEl.textContent = '正在检测...';
    
    const resultDiv = document.getElementById('conflict-result');
    resultDiv.innerHTML = '';
    resultDiv.style.display = 'none';
    
    fetch('/api/ipinfo/conflict', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({ip_range: ipRange})
    })
    .then(res => res.json())
    .then(data => {
        conflictDetectionRunning = false;
        if (data.success) {
            showConflictResult(data);
        } else {
            statusEl.textContent = `错误: ${data.error || '检测失败'}`;
        }
    })
    .catch(err => {
        conflictDetectionRunning = false;
        statusEl.textContent = `请求失败: ${err.message}`;
    });
}

function stopConflictDetection() {
    conflictDetectionRunning = false;
    document.getElementById('conflict-status').textContent = '检测已停止';
}

function showConflictResult(data) {
    const statusEl = document.getElementById('conflict-status');
    const resultDiv = document.getElementById('conflict-result');
    
    statusEl.textContent = `总计: ${data.total} | 在线: ${data.online} | 冲突: ${data.conflict}`;
    
    let html = `<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
        <thead>
            <tr style="background: #f5f7fa;">
                <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">IP地址</th>
                <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">MAC地址</th>
                <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">状态</th>
                <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">冲突次数</th>
                <th style="padding: 8px; border: 1px solid #ddd; text-align: left;">冲突MAC地址</th>
            </tr>
        </thead>
        <tbody>`;
    
    data.results.forEach(result => {
        let bgColor = '';
        let statusText = '';
        
        if (result.status === '冲突') {
            bgColor = '#ffcccc';
            statusText = '⚠️ 冲突';
        } else if (result.status === '在线') {
            bgColor = '#ccffcc';
            statusText = '✅ 在线';
        } else if (result.status === '离线') {
            bgColor = '#f0f0f0';
            statusText = '⚪ 离线';
        } else {
            bgColor = '#fff3cd';
            statusText = '❌ 错误';
        }
        
        html += `<tr style="background: ${bgColor};">
            <td style="padding: 6px 8px; border: 1px solid #ddd;">${result.ip}</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd;">${result.mac_address}</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd;">${statusText}</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd;">${result.conflict_count}</td>
            <td style="padding: 6px 8px; border: 1px solid #ddd;">${result.conflict_macs || '-'}</td>
        </tr>`;
    });
    
    html += `</tbody></table>`;
    
    resultDiv.innerHTML = html;
    resultDiv.style.display = 'block';
}

function showIPInfo(data, type) {
    const result = document.getElementById('ipinfo-result');
    let html = '';
    
    // 显示网络接口信息（仅本机IP查询时显示）
    if (data.interfaces) {
        html += `<div style="margin-bottom: 15px;"><strong style="color: #409eff;">网络接口:</strong></div>`;
        data.interfaces.forEach(iface => {
            const ipv4 = iface.ipv4 && iface.ipv4.length > 0 ? iface.ipv4[0].address : 'N/A';
            html += `<div style="margin-bottom: 8px; padding-left: 20px;">
                <strong>${iface.name}:</strong> <span style="color: #409eff;">${ipv4}</span>`;
            if (iface.mac) {
                html += ` <span style="color: #909399;">(${iface.mac})</span>`;
            }
            html += `</div>`;
        });
        if (data.default_gateway) {
            html += `<div style="margin-top: 10px;"><strong>默认网关:</strong> <span style="color: #409eff;">${data.default_gateway}</span></div>`;
        }
    }
    
    // 显示公网IP（仅公网IP查询时显示）
    if (data.public_ip) {
        html += `<div style="margin-top: 20px; margin-bottom: 15px;"><strong style="color: #409eff;">🌍 公网IP:</strong> <span style="color: #409eff; font-size: 16px;">${data.public_ip}</span></div>`;
    }
    
    // 显示地理位置信息（统一处理，避免重复）
    if (data.location) {
        // 如果是查询指定IP，先显示查询IP
        if (data.ip) {
            html += `<div style="margin-top: 15px;"><strong style="color: #409eff;">查询IP:</strong> <span style="color: #409eff; font-size: 16px;">${data.ip}</span></div>`;
        }
        
        // 显示地理位置信息（只显示一次）
        html += `<div style="margin-top: 15px;"><strong style="color: #409eff;">📍 地理位置:</strong></div>`;
        html += `<div style="padding-left: 20px; line-height: 1.8;">`;
        html += `<div><strong>国家:</strong> ${data.location.country || 'N/A'}</div>`;
        html += `<div><strong>地区:</strong> ${data.location.region || 'N/A'}</div>`;
        html += `<div><strong>城市:</strong> ${data.location.city || 'N/A'}</div>`;
        html += `<div><strong>ISP:</strong> ${data.location.isp || 'N/A'}</div>`;
        if (data.location.org) {
            html += `<div><strong>组织:</strong> ${data.location.org}</div>`;
        }
        if (data.location.timezone) {
            html += `<div><strong>时区:</strong> ${data.location.timezone}</div>`;
        }
        html += `</div>`;
    }
    
    result.innerHTML = html || '<div style="color: #909399;">暂无数据</div>';
}

function loadInterfaces() {
    // 加载DHCP接口列表
    fetch('/api/dhcp/interfaces')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const dhcpSelect = document.getElementById('dhcp-interface');
                if (dhcpSelect) {
                    dhcpSelect.innerHTML = ''; // 移除"自动选择"选项
                    data.data.forEach((iface, index) => {
                        const option = document.createElement('option');
                        option.value = iface.name;
                        option.textContent = `${iface.name} (${iface.ip})`;
                        dhcpSelect.appendChild(option);
                        // 默认选择第一个网卡
                        if (index === 0) {
                            dhcpSelect.value = iface.name;
                        }
                    });
                }
            }
        })
        .catch(err => console.error('加载DHCP接口失败:', err));
    
    // 加载数据包抓包接口列表
    fetch('/api/system/interfaces')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const packetSelect = document.getElementById('packet-interface');
                if (packetSelect) {
                    packetSelect.innerHTML = '<option value="">自动选择</option>';
                    data.data.forEach(iface => {
                        if (iface.isup && iface.name !== 'lo') {
                            const option = document.createElement('option');
                            option.value = iface.name;
                            option.textContent = iface.name;
                            packetSelect.appendChild(option);
                        }
                    });
                }
            }
        })
        .catch(err => console.error('加载数据包接口失败:', err));
}

function handleDHCP() {
    const iface = document.getElementById('dhcp-interface').value;
    const timeout = parseInt(document.getElementById('dhcp-timeout').value);
    
    const result = document.getElementById('dhcp-result');
    
    // 验证必须选择网卡
    if (!iface) {
        result.innerHTML = '<div style="color: #f56c6c;"><strong>❌ 请选择网络接口</strong></div>';
        return;
    }
    
    result.innerHTML = '<div style="color: #409eff;">正在检测DHCP服务器...</div>';
    
    fetch('/api/dhcp/detect', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({interface: iface, timeout})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showDHCPResult(data.data);
        } else {
            result.innerHTML = `<div style="color: #f56c6c;"><strong>❌ 错误: ${data.error || '检测失败'}</strong></div>`;
        }
    })
    .catch(err => {
        result.innerHTML = `<div style="color: #f56c6c;"><strong>❌ 请求失败: ${err.message}</strong></div>`;
    });
}

function showDHCPResult(data) {
    const result = document.getElementById('dhcp-result');
    let html = '';
    
    html += `<div style="margin-bottom: 15px;"><strong style="color: #409eff;">接口信息:</strong></div>`;
    html += `<div style="padding-left: 20px; line-height: 1.8; margin-bottom: 15px;">`;
    html += `<div><strong>网卡名称:</strong> ${data.interface.name}</div>`;
    html += `<div><strong>IP地址:</strong> <span style="color: #409eff;">${data.interface.ip}</span></div>`;
    html += `<div><strong>MAC地址:</strong> ${data.interface.mac}</div>`;
    html += `</div>`;
    
    if (data.result && data.result.status === 'success') {
        html += `<div style="margin-top: 20px; margin-bottom: 15px;"><strong style="color: #67c23a;">✅ DHCP服务器检测成功</strong></div>`;
        html += `<div style="padding-left: 20px; line-height: 1.8;">`;
        html += `<div><strong>DHCP服务器IP:</strong> <span style="color: #409eff; font-size: 16px;">${data.result.server_ip || data.result.options?.dhcp_server || 'N/A'}</span></div>`;
        
        // 显示MAC地址和厂商信息
        if (data.result.dhcp_server_mac) {
            html += `<div><strong>服务器MAC:</strong> <span style="color: #409eff;">${data.result.dhcp_server_mac}</span></div>`;
            if (data.result.dhcp_server_vendor) {
                html += `<div><strong>服务器厂商:</strong> ${data.result.dhcp_server_vendor}</div>`;
            }
        } else if (data.result.gateway_mac) {
            html += `<div><strong>服务器MAC:</strong> <span style="color: #909399;">N/A (服务器在不同子网)</span></div>`;
            html += `<div><strong>网关MAC:</strong> <span style="color: #409eff;">${data.result.gateway_mac}</span></div>`;
            if (data.result.gateway_vendor) {
                html += `<div><strong>网关厂商:</strong> ${data.result.gateway_vendor}</div>`;
            }
        } else {
            html += `<div><strong>服务器MAC:</strong> <span style="color: #909399;">无法获取</span></div>`;
        }
        
        html += `<div><strong>分配IP:</strong> <span style="color: #409eff;">${data.result.your_ip}</span></div>`;
        
        if (data.result.options) {
            if (data.result.options.subnet_mask) {
                html += `<div><strong>子网掩码:</strong> ${data.result.options.subnet_mask}</div>`;
            }
            if (data.result.options.router) {
                html += `<div><strong>网关:</strong> ${data.result.options.router}</div>`;
            }
            if (data.result.options.dns_servers && data.result.options.dns_servers.length > 0) {
                html += `<div><strong>DNS服务器:</strong> ${data.result.options.dns_servers.join(', ')}</div>`;
            }
            if (data.result.options.lease_time) {
                const hours = Math.floor(data.result.options.lease_time / 3600);
                const minutes = Math.floor((data.result.options.lease_time % 3600) / 60);
                html += `<div><strong>租约时间:</strong> ${hours}小时${minutes}分钟 (${data.result.options.lease_time}秒)</div>`;
            }
            if (data.result.options.domain_name) {
                html += `<div><strong>域名:</strong> ${data.result.options.domain_name}</div>`;
            }
        }
        html += `</div>`;
    } else {
        html += `<div style="margin-top: 20px; color: #f56c6c;">`;
        html += `<strong>❌ ${data.result ? data.result.message : '未检测到DHCP服务器'}</strong>`;
        html += `</div>`;
    }
    
    result.innerHTML = html || '<div style="color: #909399;">暂无数据</div>';
}

function startTFTP() {
    const port = parseInt(document.getElementById('tftp-port').value);
    const directory = document.getElementById('tftp-directory').value;
    const infoEl = document.getElementById('tftp-info');
    
    if (!port || port < 1 || port > 65535) {
        alert('请输入有效的端口号 (1-65535)');
        return;
    }
    
    if (!directory || directory.trim() === '') {
        alert('请输入根目录路径');
        return;
    }
    
    infoEl.textContent = '正在启动...';
    
    fetch('/api/services/tftp/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({port: port, directory: directory})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            document.getElementById('tftp-status').textContent = '运行中';
            document.getElementById('tftp-status').classList.add('running');
            infoEl.textContent = `✅ ${data.data.message || 'TFTP服务器已启动'} | 端口: ${port} | 目录: ${directory}`;
            infoEl.style.color = '#67c23a';
        } else {
            infoEl.textContent = `❌ ${data.error || '启动失败'}`;
            infoEl.style.color = '#f56c6c';
            alert(data.error || '启动失败');
        }
    })
    .catch(err => {
        infoEl.textContent = `❌ 请求失败: ${err.message}`;
        infoEl.style.color = '#f56c6c';
    });
}

function stopTFTP() {
    const infoEl = document.getElementById('tftp-info');
    infoEl.textContent = '正在停止...';
    
    fetch('/api/services/tftp/stop', {method: 'POST'})
    .then(res => res.json())
    .then(data => {
        document.getElementById('tftp-status').textContent = '未运行';
        document.getElementById('tftp-status').classList.remove('running');
        if (data.success) {
            infoEl.textContent = `✅ ${data.data.message || 'TFTP服务器已停止'}`;
            infoEl.style.color = '#909399';
        } else {
            infoEl.textContent = `❌ ${data.error || '停止失败'}`;
            infoEl.style.color = '#f56c6c';
        }
    })
    .catch(err => {
        infoEl.textContent = `❌ 请求失败: ${err.message}`;
        infoEl.style.color = '#f56c6c';
    });
}

function startFTP() {
    const port = parseInt(document.getElementById('ftp-port').value);
    const directory = document.getElementById('ftp-directory').value;
    const username = document.getElementById('ftp-username').value;
    const password = document.getElementById('ftp-password').value;
    const infoEl = document.getElementById('ftp-info');
    
    if (!port || port < 1 || port > 65535) {
        alert('请输入有效的端口号 (1-65535)');
        return;
    }
    
    if (!directory || directory.trim() === '') {
        alert('请输入根目录路径');
        return;
    }
    
    if (!username || username.trim() === '') {
        alert('请输入用户名');
        return;
    }
    
    infoEl.textContent = '正在启动...';
    
    fetch('/api/services/ftp/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({port: port, directory: directory, username: username, password: password})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            document.getElementById('ftp-status').textContent = '运行中';
            document.getElementById('ftp-status').classList.add('running');
            infoEl.textContent = `✅ ${data.data.message || 'FTP服务器已启动'} | 端口: ${port} | 目录: ${directory}`;
            infoEl.style.color = '#67c23a';
        } else {
            infoEl.textContent = `❌ ${data.error || data.message || '启动失败'}`;
            infoEl.style.color = '#f56c6c';
            alert(data.error || data.message || '启动失败');
        }
    })
    .catch(err => {
        infoEl.textContent = `❌ 请求失败: ${err.message}`;
        infoEl.style.color = '#f56c6c';
    });
}

function stopFTP() {
    const infoEl = document.getElementById('ftp-info');
    infoEl.textContent = '正在停止...';
    
    fetch('/api/services/ftp/stop', {method: 'POST'})
    .then(res => res.json())
    .then(data => {
        document.getElementById('ftp-status').textContent = '未运行';
        document.getElementById('ftp-status').classList.remove('running');
        if (data.success) {
            infoEl.textContent = `✅ ${data.data.message || 'FTP服务器已停止'}`;
            infoEl.style.color = '#909399';
        } else {
            infoEl.textContent = `❌ ${data.error || '停止失败'}`;
            infoEl.style.color = '#f56c6c';
        }
    })
    .catch(err => {
        infoEl.textContent = `❌ 请求失败: ${err.message}`;
        infoEl.style.color = '#f56c6c';
    });
}

function startSyslog() {
    const port = parseInt(document.getElementById('syslog-port').value);
    const logFile = document.getElementById('syslog-file').value;
    const infoEl = document.getElementById('syslog-info');
    
    if (!port || port < 1 || port > 65535) {
        alert('请输入有效的端口号 (1-65535)');
        return;
    }
    
    if (!logFile || logFile.trim() === '') {
        alert('请输入日志文件路径');
        return;
    }
    
    infoEl.textContent = '正在启动...';
    
    fetch('/api/services/syslog/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({port: port, log_file: logFile})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            document.getElementById('syslog-status').textContent = '运行中';
            document.getElementById('syslog-status').classList.add('running');
            infoEl.textContent = `✅ ${data.data.message || 'Syslog服务器已启动'} | 端口: ${port} | 日志: ${logFile}`;
            infoEl.style.color = '#67c23a';
        } else {
            infoEl.textContent = `❌ ${data.error || '启动失败'}`;
            infoEl.style.color = '#f56c6c';
            alert(data.error || '启动失败');
        }
    })
    .catch(err => {
        infoEl.textContent = `❌ 请求失败: ${err.message}`;
        infoEl.style.color = '#f56c6c';
    });
}

function stopSyslog() {
    const infoEl = document.getElementById('syslog-info');
    infoEl.textContent = '正在停止...';
    
    fetch('/api/services/syslog/stop', {method: 'POST'})
    .then(res => res.json())
    .then(data => {
        document.getElementById('syslog-status').textContent = '未运行';
        document.getElementById('syslog-status').classList.remove('running');
        if (data.success) {
            infoEl.textContent = `✅ ${data.data.message || 'Syslog服务器已停止'}`;
            infoEl.style.color = '#909399';
        } else {
            infoEl.textContent = `❌ ${data.error || '停止失败'}`;
            infoEl.style.color = '#f56c6c';
        }
    })
    .catch(err => {
        infoEl.textContent = `❌ 请求失败: ${err.message}`;
        infoEl.style.color = '#f56c6c';
    });
}

function loadSystemInterfaces() {
    fetch('/api/system/interfaces')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                showSystemInfo('interfaces', data.data);
            }
        })
        .catch(err => console.error(err));
}

function loadMACList() {
    fetch('/api/system/mac/list')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                showSystemInfo('mac', data.data);
            }
        })
        .catch(err => console.error(err));
}

function loadRoutes() {
    fetch('/api/system/routes')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                showSystemInfo('routes', data.data);
            }
        })
        .catch(err => console.error(err));
}

function showSystemInfo(type, data) {
    const result = document.getElementById('system-result');
    let html = '';
    
    if (type === 'interfaces') {
        html = '<table style="width: 100%; border-collapse: collapse;"><tr style="background: #f5f7fa;"><th style="padding: 8px; text-align: left;">接口名</th><th style="padding: 8px; text-align: left;">IP地址</th><th style="padding: 8px; text-align: left;">MAC地址</th><th style="padding: 8px; text-align: left;">状态</th></tr>';
        data.forEach(iface => {
            html += `<tr><td style="padding: 8px; border-bottom: 1px solid #eee;">${iface.name}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${iface.ipv4 || 'N/A'}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${iface.mac || 'N/A'}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${iface.isup ? '启用' : '禁用'}</td></tr>`;
        });
        html += '</table>';
    } else if (type === 'mac') {
        html = '<table style="width: 100%; border-collapse: collapse;"><tr style="background: #f5f7fa;"><th style="padding: 8px; text-align: left;">接口</th><th style="padding: 8px; text-align: left;">MAC地址</th><th style="padding: 8px; text-align: left;">厂商</th></tr>';
        data.forEach(item => {
            html += `<tr><td style="padding: 8px; border-bottom: 1px solid #eee;">${item.interface}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${item.mac}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${item.vendor || 'Unknown'}</td></tr>`;
        });
        html += '</table>';
    } else if (type === 'routes') {
        html = '<table style="width: 100%; border-collapse: collapse;"><tr style="background: #f5f7fa;"><th style="padding: 8px; text-align: left;">目标</th><th style="padding: 8px; text-align: left;">网关</th><th style="padding: 8px; text-align: left;">接口</th></tr>';
        data.forEach(route => {
            html += `<tr><td style="padding: 8px; border-bottom: 1px solid #eee;">${route.destination || 'default'}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${route.gateway || 'N/A'}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${route.interface || 'N/A'}</td></tr>`;
        });
        html += '</table>';
    }
    
    result.innerHTML = html;
}

function handleSpeedTest() {
    const type = document.getElementById('speedtest-type').value;
    const result = document.getElementById('speedtest-result');
    result.textContent = '正在测试网速，请稍候...';
    
    fetch('/api/speedtest/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({test_type: type})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            let text = `测试类型: ${type}\n\n`;
            text += `延迟: ${data.data.latency}ms\n`;
            if (data.data.download_speed) {
                text += `下载速度: ${data.data.download_speed} Mbps\n`;
            }
            if (data.data.upload_speed) {
                text += `上传速度: ${data.data.upload_speed} Mbps\n`;
            }
            result.textContent = text;
        } else {
            result.textContent = `错误: ${data.error || '测试失败'}`;
        }
    })
    .catch(err => {
        result.textContent = `请求失败: ${err.message}`;
    });
}

function stopSpeedTest() {
    fetch('/api/speedtest/stop', {method: 'POST'});
}

function scanWiFi() {
    const result = document.getElementById('wifi-result');
    result.textContent = '正在扫描WiFi网络...';
    
    fetch('/api/wifi/scan')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                showWiFiResult(data.data);
            } else {
                result.textContent = `错误: ${data.error || '扫描失败'}`;
            }
        })
        .catch(err => {
            result.textContent = `请求失败: ${err.message}`;
        });
}

function getWiFiPasswords() {
    const result = document.getElementById('wifi-result');
    result.textContent = '正在获取WiFi密码...';
    
    fetch('/api/wifi/passwords')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                showWiFiPasswords(data.data);
            } else {
                result.textContent = `错误: ${data.error || '获取失败'}`;
            }
        })
        .catch(err => {
            result.textContent = `请求失败: ${err.message}`;
        });
}

function showWiFiResult(data) {
    const result = document.getElementById('wifi-result');
    if (data.length === 0) {
        result.textContent = '未发现WiFi网络';
        return;
    }
    
    let html = '<table style="width: 100%; border-collapse: collapse;"><tr style="background: #f5f7fa;"><th style="padding: 8px; text-align: left;">SSID</th><th style="padding: 8px; text-align: left;">信号强度</th><th style="padding: 8px; text-align: left;">加密</th><th style="padding: 8px; text-align: left;">频率</th></tr>';
    data.forEach(net => {
        html += `<tr><td style="padding: 8px; border-bottom: 1px solid #eee;">${net.ssid || 'Unknown'}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${net.signal || 'N/A'}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${net.security || 'N/A'}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${net.frequency || 'N/A'}</td></tr>`;
    });
    html += '</table>';
    result.innerHTML = html;
}

function showWiFiPasswords(data) {
    const result = document.getElementById('wifi-result');
    if (data.length === 0) {
        result.textContent = '未找到保存的WiFi密码';
        return;
    }
    
    let html = '<table style="width: 100%; border-collapse: collapse;"><tr style="background: #f5f7fa;"><th style="padding: 8px; text-align: left;">SSID</th><th style="padding: 8px; text-align: left;">密码</th></tr>';
    data.forEach(item => {
        html += `<tr><td style="padding: 8px; border-bottom: 1px solid #eee;">${item.ssid}</td><td style="padding: 8px; border-bottom: 1px solid #eee;">${item.password}</td></tr>`;
    });
    html += '</table>';
    result.innerHTML = html;
}

// 旧的抓包函数已删除，使用下面的新实现

function formatBytes(bytes) {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

// 环路检测功能
function loadLoopInterfaces() {
    fetch('/api/loop/interfaces')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const select = document.getElementById('loop-interface');
                select.innerHTML = '<option value="">全部网卡</option>';
                data.interfaces.forEach(iface => {
                    const option = document.createElement('option');
                    option.value = iface.name;
                    const displayName = iface.ip ? `${iface.name} (${iface.ip})` : `${iface.name} (无IP)`;
                    option.textContent = displayName;
                    select.appendChild(option);
                });
            }
        })
        .catch(err => {
            console.error('加载接口列表失败:', err);
        });
}

function startLoopDetection() {
    const interfaceName = document.getElementById('loop-interface').value || null;
    const result = document.getElementById('loop-result');
    
    result.textContent = '🔗 开始网络环路检测...\n\n';
    
    fetch('/api/loop/detect', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({interface_name: interfaceName})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            showLoopResult(data.result, data.analysis);
        } else {
            result.textContent = `❌ 检测失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

function showLoopResult(result, analysis) {
    const resultEl = document.getElementById('loop-result');
    const now = new Date();
    const timeStr = now.toLocaleString('zh-CN');
    
    let report = `🔗 网络环路检测结果\n`;
    report += `${'═'.repeat(50)}\n\n`;
    
    // 显示检测的网卡信息
    if (result.interface_name) {
        report += `检测网卡: ${result.interface_name}\n`;
    } else {
        report += `检测范围: 全部网卡\n`;
    }
    report += `检测时间: ${timeStr}\n\n`;
    
    // 检测结果
    report += `检测方法: 以太网帧检测（底层检测）\n`;
    report += `检测状态: ${result.status}\n`;
    report += `综合评分: ${analysis.total_score.toFixed(2)}/5.0 分\n`;
    report += `风险等级: ${analysis.risk_level}\n\n`;
    
    // 详细信息 - 分行显示
    report += `详细信息:\n`;
    if (result.details) {
        // 如果包含分号，说明是多个网卡的结果
        if (result.details.includes('；')) {
            const details = result.details.split('；');
            details.forEach((detail, index) => {
                if (detail.trim()) {
                    report += `  ${index + 1}. ${detail.trim()}\n`;
                }
            });
        } else {
            // 单个网卡或没有分号的情况
            report += `  ${result.details}\n`;
        }
    }
    report += `\n`;
    
    // 建议
    report += `建议: ${analysis.suggestion}\n`;
    
    resultEl.textContent = report;
}

function clearLoopResult() {
    const result = document.getElementById('loop-result');
    result.textContent = `╔══════════════════════════════════════════════════════════════════╗
║                    网络环路检测工具                              ║
╚══════════════════════════════════════════════════════════════════╝

👋 欢迎使用网络环路检测功能！

本工具使用以下方法检测网络环路：

🔌 以太网帧检测（底层检测）
   • 通过监控非单播流量变化进行底层检测
   • 检测网络环路最准确的方法

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

💡 什么是网络环路？

网络环路是指网络中存在多条路径形成闭环，导致数据包
在网络中无限循环，造成网络拥塞甚至瘫痪。

常见原因：
• 交换机端口错误连接
• 网线误接形成环路
• 未启用STP协议

点击上方"开始环路检测"按钮开始检测...`;
}

// 摄像头扫描功能
let cameraScanInterval = null;

function startCameraScan() {
    const network = document.getElementById('camera-network').value.trim();
    const brands = Array.from(document.querySelectorAll('.camera-brand:checked')).map(cb => cb.value);
    const scanMode = document.querySelector('input[name="camera-mode"]:checked').value;
    const maxThreads = parseInt(document.getElementById('camera-threads').value);
    const timeout = parseFloat(document.getElementById('camera-timeout').value);
    
    if (!network) {
        alert('请输入网络范围');
        return;
    }
    
    if (brands.length === 0) {
        alert('请至少选择一个摄像头品牌');
        return;
    }
    
    const result = document.getElementById('camera-result');
    result.textContent = '🔍 开始扫描摄像头...\n\n';
    
    document.getElementById('camera-stop-btn').disabled = false;
    
    fetch('/api/camera/scan/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({
            network,
            brands,
            scan_mode: scanMode,
            max_threads: maxThreads,
            timeout
        })
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            if (cameraScanInterval) {
                clearInterval(cameraScanInterval);
            }
            cameraScanInterval = setInterval(updateCameraStatus, 1000);
            updateCameraStatus();
        } else {
            result.textContent = `❌ 扫描失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

function stopCameraScan() {
    fetch('/api/camera/scan/stop', {method: 'POST'})
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                if (cameraScanInterval) {
                    clearInterval(cameraScanInterval);
                    cameraScanInterval = null;
                }
                document.getElementById('camera-stop-btn').disabled = true;
            }
        });
}

function updateCameraStatus() {
    fetch('/api/camera/scan/status')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                document.getElementById('camera-total').textContent = data.total || 0;
                document.getElementById('camera-scanned').textContent = data.scanned || 0;
                document.getElementById('camera-found').textContent = data.cameras ? data.cameras.length : 0;
                
                const progress = data.total > 0 ? ((data.scanned / data.total) * 100).toFixed(1) : 0;
                document.getElementById('camera-progress').textContent = progress + '%';
                
                const result = document.getElementById('camera-result');
                const tableContainer = document.getElementById('camera-table-container');
                const tableBody = document.getElementById('camera-table-body');
                
                if (data.running) {
                    result.style.display = 'block';
                    tableContainer.style.display = 'none';
                    result.textContent = `🔍 扫描中... (已检测 ${data.scanned}/${data.total})\n\n请稍候...`;
                } else {
                    if (cameraScanInterval) {
                        clearInterval(cameraScanInterval);
                        cameraScanInterval = null;
                    }
                    document.getElementById('camera-stop-btn').disabled = true;
                    
                    if (data.cameras && data.cameras.length > 0) {
                        // 显示表格
                        result.style.display = 'none';
                        tableContainer.style.display = 'block';
                        
                        // 清空表格
                        tableBody.innerHTML = '';
                        
                        // 填充表格数据
                        data.cameras.forEach((cam, idx) => {
                            const row = document.createElement('tr');
                            row.style.borderBottom = '1px solid #eee';
                            row.innerHTML = `
                                <td style="padding: 8px;">${idx + 1}</td>
                                <td style="padding: 8px;">
                                    <span style="font-family: monospace;">${cam.ip}</span>
                                    <button class="btn btn-sm" onclick="copyText('${cam.ip}')" style="padding: 2px 6px; font-size: 11px; margin-left: 5px;" title="复制IP">
                                        <i class="fas fa-copy"></i>
                                    </button>
                                </td>
                                <td style="padding: 8px;">${cam.port}</td>
                                <td style="padding: 8px;">${cam.brand}</td>
                                <td style="padding: 8px;">${cam.model || '-'}</td>
                                <td style="padding: 8px; font-family: monospace; font-size: 12px;">${cam.mac || '-'}</td>
                                <td style="padding: 8px;">${cam.response_time || '-'}</td>
                                <td style="padding: 8px;">
                                    <button class="btn btn-sm btn-primary" onclick="openCameraWeb('${cam.web_url}')" style="padding: 4px 8px; font-size: 11px; margin-right: 5px;" title="打开Web界面">
                                        <i class="fas fa-external-link-alt"></i>
                                    </button>
                                    <button class="btn btn-sm btn-default" onclick="recheckCamera('${cam.ip}', ${cam.port})" style="padding: 4px 8px; font-size: 11px;" title="重新检测">
                                        <i class="fas fa-sync"></i>
                                    </button>
                                </td>
                            `;
                            tableBody.appendChild(row);
                        });
                    } else {
                        // 显示文本结果
                        result.style.display = 'block';
                        tableContainer.style.display = 'none';
                        result.textContent = `🔍 扫描完成！\n\n未发现摄像头设备`;
                    }
                }
                
                if (data.error) {
                    result.style.display = 'block';
                    tableContainer.style.display = 'none';
                    result.textContent = `❌ 扫描出错: ${data.error}`;
                }
            }
        })
        .catch(err => {
            console.error('获取扫描状态失败:', err);
        });
}

function clearCameraResult() {
    const result = document.getElementById('camera-result');
    const tableContainer = document.getElementById('camera-table-container');
    const tableBody = document.getElementById('camera-table-body');
    
    result.style.display = 'block';
    tableContainer.style.display = 'none';
    result.textContent = `📹 摄像头扫描工具

本工具支持扫描以下品牌的监控摄像头：
• 海康威视
• 大华
• 宇视
• 天地伟业
• 通用摄像头

扫描模式：
• 快速扫描：扫描最常用端口
• 深度扫描：扫描更多端口，识别率更高
• ONVIF发现：基于ONVIF标准协议发现

准备就绪，点击开始扫描...`;
    
    tableBody.innerHTML = '';
    document.getElementById('camera-total').textContent = '0';
    document.getElementById('camera-scanned').textContent = '0';
    document.getElementById('camera-found').textContent = '0';
    document.getElementById('camera-progress').textContent = '0%';
}

// 摄像头扫描专业功能
function exportCameraResult() {
    fetch('/api/camera/scan/status')
        .then(res => res.json())
        .then(data => {
            if (data.success && data.cameras && data.cameras.length > 0) {
                let content = `摄像头扫描结果报告\n`;
                content += `${'═'.repeat(60)}\n`;
                content += `扫描时间: ${new Date().toLocaleString('zh-CN')}\n`;
                content += `扫描IP数: ${data.total}\n`;
                content += `已检测: ${data.scanned}\n`;
                content += `发现摄像头: ${data.cameras.length}\n`;
                content += `${'═'.repeat(60)}\n\n`;
                
                data.cameras.forEach((cam, idx) => {
                    content += `${idx + 1}. IP地址: ${cam.ip}\n`;
                    content += `   端口: ${cam.port}\n`;
                    content += `   品牌: ${cam.brand}\n`;
                    content += `   型号: ${cam.model || '-'}\n`;
                    content += `   MAC地址: ${cam.mac || '-'}\n`;
                    content += `   Web地址: ${cam.web_url}\n`;
                    content += `   响应时间: ${cam.response_time || '-'}\n`;
                    content += `\n`;
                });
                
                // 使用Blob API下载文件
                const blob = new Blob([content], { type: 'text/plain;charset=utf-8' });
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `camera_scan_${new Date().getTime()}.txt`;
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                URL.revokeObjectURL(url);
                
                alert('结果已导出！');
            } else {
                alert('没有可导出的结果');
            }
        })
        .catch(err => {
            alert('导出失败: ' + err.message);
        });
}

function copyAllCameraIPs() {
    fetch('/api/camera/scan/status')
        .then(res => res.json())
        .then(data => {
            if (data.success && data.cameras && data.cameras.length > 0) {
                const ips = data.cameras.map(cam => cam.ip).join('\n');
                copyText(ips);
                alert(`已复制 ${data.cameras.length} 个IP地址到剪贴板`);
            } else {
                alert('没有可复制的IP地址');
            }
        });
}

function copyText(text) {
    if (navigator.clipboard) {
        navigator.clipboard.writeText(text).then(() => {
            console.log('已复制到剪贴板');
        });
    } else {
        // 降级方案
        const textarea = document.createElement('textarea');
        textarea.value = text;
        textarea.style.position = 'fixed';
        textarea.style.opacity = '0';
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
    }
}

function openCameraWeb(url) {
    window.open(url, '_blank');
}

function recheckCamera(ip, port) {
    const brands = Array.from(document.querySelectorAll('.camera-brand:checked')).map(cb => cb.value);
    const scanMode = document.querySelector('input[name="camera-mode"]:checked').value;
    const timeout = parseFloat(document.getElementById('camera-timeout').value);
    
    fetch('/api/camera/recheck', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({
            ip: ip,
            port: port,
            brands: brands,
            scan_mode: scanMode,
            timeout: timeout
        })
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            alert(`重新检测完成！\n品牌: ${data.camera.brand}\n型号: ${data.camera.model || '-'}`);
            // 刷新状态
            updateCameraStatus();
        } else {
            alert('重新检测失败: ' + (data.error || '未知错误'));
        }
    })
    .catch(err => {
        alert('请求失败: ' + err.message);
    });
}

// 会话数测试功能
let sessionTestInterval = null;

function startSessionTest() {
    console.log('开始测试被调用');
    
    const serverRadio = document.querySelector('input[name="session-server"]:checked');
    if (!serverRadio) {
        alert('请选择测试服务器');
        return;
    }
    const serverType = serverRadio.value;
    const targetHost = document.getElementById('session-target-ip') ? document.getElementById('session-target-ip').value.trim() : '';
    const targetPort = parseInt(document.getElementById('session-target-port').value) || 80;
    const delay = parseInt(document.getElementById('session-delay').value) || 100;
    const threadCount = parseInt(document.getElementById('session-threads').value) || 10;
    const maxConn = parseInt(document.getElementById('session-max-conn').value) || 10000;
    const maxFail = parseInt(document.getElementById('session-max-fail').value) || 100;
    
    if (serverType === 'custom' && !targetHost) {
        alert('请输入目标IP地址或域名');
        return;
    }
    
    const log = document.getElementById('session-log');
    if (!log) {
        console.error('找不到session-log元素');
        alert('找不到日志显示区域');
        return;
    }
    log.textContent = '开始会话数测试...\n';
    
    const stopBtn = document.getElementById('session-stop-btn');
    if (stopBtn) {
        stopBtn.disabled = false;
    }
    
    console.log('发送请求:', {server_type: serverType, target_host: targetHost, target_port: targetPort});
    
    fetch('/api/session/test/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({
            server_type: serverType,
            target_host: targetHost,
            target_port: targetPort,
            delay: delay,
            thread_count: threadCount,
            max_conn: maxConn,
            max_fail: maxFail
        })
    })
    .then(res => {
        console.log('响应状态:', res.status);
        return res.json();
    })
    .then(data => {
        console.log('响应数据:', data);
        if (data.success) {
            // 开始轮询状态
            if (sessionTestInterval) {
                clearInterval(sessionTestInterval);
            }
            sessionTestInterval = setInterval(updateSessionStatus, 500);
            updateSessionStatus();
        } else {
            log.textContent = `❌ 测试失败: ${data.error || '未知错误'}`;
            if (stopBtn) {
                stopBtn.disabled = true;
            }
        }
    })
    .catch(err => {
        console.error('请求错误:', err);
        log.textContent = `❌ 请求失败: ${err.message}`;
        if (stopBtn) {
            stopBtn.disabled = true;
        }
    });
}

function stopSessionTest() {
    console.log('停止测试被调用');
    fetch('/api/session/test/stop', {method: 'POST'})
        .then(res => {
            console.log('停止测试响应状态:', res.status);
            return res.json();
        })
        .then(data => {
            console.log('停止测试响应数据:', data);
            if (data.success) {
                if (sessionTestInterval) {
                    clearInterval(sessionTestInterval);
                    sessionTestInterval = null;
                }
                const stopBtn = document.getElementById('session-stop-btn');
                if (stopBtn) {
                    stopBtn.disabled = true;
                }
                const log = document.getElementById('session-log');
                if (log) {
                    log.textContent += '\n测试已停止\n';
                }
            } else {
                alert('停止测试失败: ' + (data.error || '未知错误'));
            }
        })
        .catch(err => {
            console.error('停止测试请求失败:', err);
            alert('请求失败: ' + err.message);
        });
}

function updateSessionStatus() {
    fetch('/api/session/test/status')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                // 更新统计信息
                const successEl = document.getElementById('session-success');
                const failEl = document.getElementById('session-fail');
                const timeEl = document.getElementById('session-time');
                const activeEl = document.getElementById('session-active') || document.getElementById('session-active-conn');
                
                if (successEl) successEl.textContent = data.stats.success || 0;
                if (failEl) failEl.textContent = data.stats.fail || 0;
                if (timeEl) timeEl.textContent = (data.stats.time || 0) + 's';
                if (activeEl) activeEl.textContent = data.stats.active || 0;
                
                // 更新日志
                const log = document.getElementById('session-log');
                if (data.logs && data.logs.length > 0) {
                    let logText = '';
                    data.logs.forEach(logEntry => {
                        const time = new Date(logEntry.time * 1000).toLocaleTimeString('zh-CN');
                        logText += `[${time}] ${logEntry.message}\n`;
                    });
                    log.textContent = logText;
                    log.scrollTop = log.scrollHeight;
                }
                
                // 如果测试结束，停止轮询
                if (!data.running) {
                    if (sessionTestInterval) {
                        clearInterval(sessionTestInterval);
                        sessionTestInterval = null;
                    }
                    document.getElementById('session-stop-btn').disabled = true;
                }
            }
        })
        .catch(err => {
            console.error('获取测试状态失败:', err);
        });
}

// 数据包抓包功能
let packetCaptureInterval = null;

function loadPacketInterfaces() {
    fetch('/api/packet/interfaces')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const select = document.getElementById('packet-interface');
                if (select) {
                    select.innerHTML = '<option value="">自动选择</option>';
                    data.interfaces.forEach(iface => {
                        const option = document.createElement('option');
                        option.value = iface.name;
                        option.textContent = `${iface.name} (${iface.ip})`;
                        select.appendChild(option);
                    });
                }
            }
        })
        .catch(err => console.error('加载接口失败:', err));
}

function startPacketCapture() {
    const interface = document.getElementById('packet-interface').value;
    const filter = document.getElementById('packet-filter').value.trim();
    const count = parseInt(document.getElementById('packet-count').value) || 100;
    
    const result = document.getElementById('packet-result');
    const stats = document.getElementById('packet-stats');
    
    if (result) {
        result.textContent = '开始抓包...\n';
    }
    if (stats) {
        stats.textContent = '正在抓包...';
    }
    
    fetch('/api/packet/start', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({
            interface: interface,
            filter: filter,
            count: count
        })
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            // 开始轮询状态
            if (packetCaptureInterval) {
                clearInterval(packetCaptureInterval);
            }
            packetCaptureInterval = setInterval(updatePacketStatus, 500);
            updatePacketStatus();
        } else {
            if (result) {
                result.textContent = `❌ 抓包失败: ${data.error || '未知错误'}`;
            }
            if (stats) {
                stats.textContent = '抓包失败';
            }
        }
    })
    .catch(err => {
        console.error('抓包请求失败:', err);
        if (result) {
            result.textContent = `❌ 请求失败: ${err.message}`;
        }
        if (stats) {
            stats.textContent = '请求失败';
        }
    });
}

function stopPacketCapture() {
    fetch('/api/packet/stop', {method: 'POST'})
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                if (packetCaptureInterval) {
                    clearInterval(packetCaptureInterval);
                    packetCaptureInterval = null;
                }
                const stats = document.getElementById('packet-stats');
                if (stats) {
                    stats.textContent = '抓包已停止';
                }
            } else {
                alert('停止抓包失败: ' + (data.error || '未知错误'));
            }
        })
        .catch(err => {
            console.error('停止抓包请求失败:', err);
            alert('请求失败: ' + err.message);
        });
}

function updatePacketStatus() {
    fetch('/api/packet/status')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                // 更新统计信息
                const stats = document.getElementById('packet-stats');
                if (stats) {
                    if (data.running) {
                        stats.textContent = `已捕获 ${data.stats.total} 个数据包 | TCP: ${data.stats.tcp} | UDP: ${data.stats.udp} | ICMP: ${data.stats.icmp} | 其他: ${data.stats.other}`;
                    } else {
                        stats.textContent = `抓包已停止，共捕获 ${data.stats.total} 个数据包 | TCP: ${data.stats.tcp} | UDP: ${data.stats.udp} | ICMP: ${data.stats.icmp} | 其他: ${data.stats.other}`;
                    }
                }
                
                // 更新日志
                const log = document.getElementById('packet-result');
                if (log && data.logs && data.logs.length > 0) {
                    let logText = '';
                    data.logs.forEach(logEntry => {
                        const time = new Date(logEntry.time * 1000).toLocaleTimeString('zh-CN');
                        logText += `[${time}] ${logEntry.message}\n`;
                    });
                    logText += '\n';
                    log.textContent = logText;
                }
                
                // 更新数据包显示
                if (log && data.packets && data.packets.length > 0) {
                    let packetText = '';
                    // 只显示最近50个包
                    const displayPackets = data.packets.slice(-50);
                    displayPackets.forEach(packet => {
                        const timestamp = new Date(packet.timestamp * 1000).toLocaleTimeString('zh-CN', {hour12: false, fractionalSecondDigits: 3});
                        packetText += `[${timestamp}] `;
                        packetText += `${packet.src_ip || 'N/A'}:${packet.src_port || 'N/A'} -> `;
                        packetText += `${packet.dst_ip || 'N/A'}:${packet.dst_port || 'N/A'} `;
                        packetText += `[${packet.transport || 'N/A'}] `;
                        packetText += `Length: ${packet.length || 'N/A'} `;
                        if (packet.ttl) {
                            packetText += `TTL: ${packet.ttl} `;
                        }
                        if (packet.transport === 'TCP' && packet.flags) {
                            packetText += `Flags: ${packet.flags}`;
                        } else if (packet.transport === 'ICMP' && packet.icmp_type !== undefined) {
                            packetText += `Type: ${packet.icmp_type} Code: ${packet.icmp_code}`;
                        }
                        packetText += '\n';
                    });
                    
                    // 如果有日志，先显示日志，再显示数据包
                    if (data.logs && data.logs.length > 0) {
                        let logText = '';
                        data.logs.forEach(logEntry => {
                            const time = new Date(logEntry.time * 1000).toLocaleTimeString('zh-CN');
                            logText += `[${time}] ${logEntry.message}\n`;
                        });
                        logText += '\n' + '='.repeat(60) + '\n\n';
                        log.textContent = logText + packetText;
                    } else {
                        log.textContent = packetText;
                    }
                    
                    log.scrollTop = log.scrollHeight;
                }
                
                // 如果抓包结束，停止轮询
                if (!data.running) {
                    if (packetCaptureInterval) {
                        clearInterval(packetCaptureInterval);
                        packetCaptureInterval = null;
                    }
                }
            }
        })
        .catch(err => {
            console.error('获取抓包状态失败:', err);
        });
}

function exportPacketPcap() {
    // 检查是否有数据包
    fetch('/api/packet/status')
        .then(res => res.json())
        .then(data => {
            if (!data.success) {
                alert('获取抓包状态失败: ' + (data.error || '未知错误'));
                return;
            }
            
            if (!data.packets || data.packets.length === 0) {
                alert('没有可导出的数据包！\n请先进行抓包操作。');
                return;
            }
            
            // 导出PCAP文件
            fetch('/api/packet/export', {method: 'POST'})
                .then(res => {
                    if (!res.ok) {
                        return res.json().then(err => Promise.reject(err));
                    }
                    return res.blob();
                })
                .then(blob => {
                    // 创建下载链接
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
                    a.href = url;
                    a.download = `capture_${timestamp}.pcap`;
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);
                    
                    alert(`✅ 已成功导出 ${data.packets.length} 个数据包\n\n💡 提示:\n• 可使用 Wireshark 打开此文件进行深度分析\n• 可使用 tcpdump 或其他工具读取\n• PCAP 格式是通用的网络数据包格式`);
                })
                .catch(err => {
                    console.error('导出失败:', err);
                    alert('导出失败: ' + (err.error || err.message || '未知错误'));
                });
        })
        .catch(err => {
            console.error('获取状态失败:', err);
            alert('获取抓包状态失败: ' + err.message);
        });
}

function clearPacketResult() {
    fetch('/api/packet/clear', {method: 'POST'})
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const result = document.getElementById('packet-result');
                const stats = document.getElementById('packet-stats');
                if (result) {
                    result.textContent = '已清空';
                }
                if (stats) {
                    stats.textContent = '准备就绪';
                }
            }
        })
        .catch(err => {
            console.error('清空失败:', err);
        });
}

function clearSessionLog() {
    const log = document.getElementById('session-log');
    log.textContent = '准备就绪，点击开始测试...';
    
    document.getElementById('session-success').textContent = '0';
    document.getElementById('session-fail').textContent = '0';
    document.getElementById('session-time').textContent = '0s';
    document.getElementById('session-active').textContent = '0';
}

// 关于页面功能
function switchAboutTab(tab) {
    // 隐藏所有标签页内容
    document.querySelectorAll('.about-tab-content').forEach(content => {
        content.style.display = 'none';
    });
    
    // 移除所有按钮的active类
    const tabButtons = document.querySelectorAll('#page-about .tab-btn');
    tabButtons.forEach(btn => {
        btn.classList.remove('active');
    });
    
    // 显示选中的标签页
    const targetContent = document.getElementById(`about-tab-${tab}`);
    if (targetContent) {
        targetContent.style.display = 'block';
    }
    
    // 激活对应的按钮
    if (tab === 'support' && tabButtons[0]) {
        tabButtons[0].classList.add('active');
    } else if (tab === 'about' && tabButtons[1]) {
        tabButtons[1].classList.add('active');
    }
}

// 更新授权状态显示
function updateLicenseStatus() {
    fetch('/api/license/status')
        .then(res => res.json())
        .then(data => {
            const statusElement = document.getElementById('license-status');
            if (statusElement) {
                if (data.success && data.is_activated) {
                    statusElement.textContent = '✅ 已激活';
                    statusElement.style.color = '#4CAF50';
                } else {
                    statusElement.textContent = '❌ 未激活';
                    statusElement.style.color = '#F44336';
                }
            }
        })
        .catch(err => {
            console.error('更新授权状态失败:', err);
        });
}

function openActivationWindow() {
    // 显示模态窗口
    const modal = document.getElementById('activation-modal');
    modal.style.display = 'flex';
    modal.classList.add('show');
    
    // 先显示加载状态
    const hwidInput = document.getElementById('hwid-display');
    if (hwidInput) {
        hwidInput.value = '正在获取...';
    }
    
    // 获取硬件码和状态
    fetch('/api/license/status')
        .then(res => {
            if (!res.ok) {
                throw new Error(`HTTP ${res.status}`);
            }
            return res.json();
        })
        .then(data => {
            const hwid = data.hwid || '获取失败';
            const isActivated = data.is_activated || false;
            
            // 更新硬件码显示
            if (hwidInput) {
                if (hwid === '获取失败' || hwid === 'ERROR-GENERATING-HWID') {
                    hwidInput.value = '获取失败，请检查系统权限或联系技术支持';
                    hwidInput.style.color = '#f44336';
                } else {
                    hwidInput.value = hwid;
                    hwidInput.style.color = '#333';
                }
            }
            
            // 更新激活状态
            const statusText = document.getElementById('activation-status-text');
            const statusDisplay = document.getElementById('activation-status-display');
            if (statusText && statusDisplay) {
                if (isActivated) {
                    statusText.textContent = '✅ 已激活';
                    statusDisplay.style.background = '#e8f5e9';
                    statusDisplay.style.borderLeftColor = '#4CAF50';
                    statusText.style.color = '#2e7d32';
                } else {
                    statusText.textContent = '❌ 未激活';
                    statusDisplay.style.background = '#ffebee';
                    statusDisplay.style.borderLeftColor = '#f44336';
                    statusText.style.color = '#c62828';
                }
            }
            
            // 如果有错误信息，显示警告
            if (data.error && !data.success) {
                console.warn('授权状态获取警告:', data.error);
            }
        })
        .catch(err => {
            console.error('获取授权状态失败:', err);
            if (hwidInput) {
                hwidInput.value = '获取失败: ' + err.message + '，请刷新页面重试';
                hwidInput.style.color = '#f44336';
            }
        });
}

function closeActivationModal() {
    const modal = document.getElementById('activation-modal');
    modal.classList.remove('show');
    setTimeout(() => {
        modal.style.display = 'none';
    }, 300);
    
    // 重置复制成功提示
    const copySuccess = document.getElementById('copy-success');
    if (copySuccess) {
        copySuccess.style.display = 'none';
    }
    
    // 重置文件选择
    const fileInput = document.getElementById('license-file-input');
    if (fileInput) {
        fileInput.value = '';
    }
    const importBtn = document.getElementById('import-license-btn');
    if (importBtn) {
        importBtn.disabled = true;
    }
    const importResult = document.getElementById('import-result');
    if (importResult) {
        importResult.textContent = '';
        importResult.className = '';
    }
}

function copyHWID() {
    const hwidInput = document.getElementById('hwid-display');
    if (!hwidInput) return;
    
    const hwid = hwidInput.value;
    if (!hwid || hwid === '正在获取...' || hwid === '获取失败，请刷新重试') {
        return;
    }
    
    // 复制到剪贴板
    hwidInput.select();
    hwidInput.setSelectionRange(0, 99999); // 移动端支持
    
    try {
        document.execCommand('copy');
        // 显示成功提示
        const copySuccess = document.getElementById('copy-success');
        if (copySuccess) {
            copySuccess.style.display = 'block';
            setTimeout(() => {
                copySuccess.style.display = 'none';
            }, 2000);
        }
        
        // 如果支持新的Clipboard API，也尝试使用
        if (navigator.clipboard) {
            navigator.clipboard.writeText(hwid).catch(err => {
                console.error('复制失败:', err);
            });
        }
    } catch (err) {
        console.error('复制失败:', err);
        alert('复制失败，请手动选择并复制');
    }
}

let selectedLicenseFile = null;

function handleLicenseFileSelect(event) {
    const file = event.target.files[0];
    if (file) {
        selectedLicenseFile = file;
        const importBtn = document.getElementById('import-license-btn');
        if (importBtn) {
            importBtn.disabled = false;
        }
    }
}

// 在线激活授权
function activateLicenseOnline() {
    const activateBtn = document.getElementById('activate-online-btn');
    const activateResult = document.getElementById('activate-result');
    
    if (activateBtn) {
        activateBtn.disabled = true;
        activateBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> 激活中...';
    }
    
    if (activateResult) {
        activateResult.textContent = '正在连接服务器...';
        activateResult.style.color = '#2196F3';
    }
    
    // 发送在线激活请求
    fetch('/api/license/activate', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'}
    })
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                if (activateResult) {
                    activateResult.textContent = '✅ 授权激活成功！';
                    activateResult.style.color = '#4CAF50';
                }
                
                // 更新状态
                updateLicenseStatus();
                
                // 2秒后关闭窗口并刷新页面
                setTimeout(() => {
                    closeActivationModal();
                    location.reload();
                }, 2000);
            } else {
                if (activateResult) {
                    activateResult.textContent = '❌ 激活失败：' + (data.error || '未知错误');
                    activateResult.style.color = '#f44336';
                }
                if (activateBtn) {
                    activateBtn.disabled = false;
                    activateBtn.innerHTML = '<i class="fas fa-key"></i> 在线激活授权';
                }
            }
        })
        .catch(err => {
            if (activateResult) {
                activateResult.textContent = '❌ 激活失败：' + err.message;
                activateResult.style.color = '#f44336';
            }
            if (activateBtn) {
                activateBtn.disabled = false;
                activateBtn.innerHTML = '<i class="fas fa-key"></i> 在线激活授权';
            }
        });
}

function importLicenseFile() {
    if (!selectedLicenseFile) {
        return;
    }
    
    const importBtn = document.getElementById('import-license-btn');
    const importResult = document.getElementById('import-result');
    
    if (importBtn) {
        importBtn.disabled = true;
        importBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> 导入中...';
    }
    
    const reader = new FileReader();
    reader.onload = function(e) {
        try {
            const licenseData = JSON.parse(e.target.result);
            
            // 发送到服务器导入
            fetch('/api/license/import', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(licenseData)
            })
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        if (importResult) {
                            importResult.textContent = '✅ 授权文件导入成功！';
                            importResult.style.color = '#4CAF50';
                        }
                        
                        // 更新状态
                        updateLicenseStatus();
                        
                        // 2秒后关闭窗口
                        setTimeout(() => {
                            closeActivationModal();
                            // 刷新页面以应用授权
                            location.reload();
                        }, 2000);
                    } else {
                        if (importResult) {
                            importResult.textContent = '❌ 导入失败：' + (data.error || '未知错误');
                            importResult.style.color = '#f44336';
                        }
                        if (importBtn) {
                            importBtn.disabled = false;
                            importBtn.innerHTML = '<i class="fas fa-upload"></i> 导入';
                        }
                    }
                })
                .catch(err => {
                    if (importResult) {
                        importResult.textContent = '❌ 导入失败：' + err.message;
                        importResult.style.color = '#f44336';
                    }
                    if (importBtn) {
                        importBtn.disabled = false;
                        importBtn.innerHTML = '<i class="fas fa-upload"></i> 导入';
                    }
                });
        } catch (err) {
            if (importResult) {
                importResult.textContent = '❌ 文件格式错误：' + err.message;
                importResult.style.color = '#f44336';
            }
            if (importBtn) {
                importBtn.disabled = false;
                importBtn.innerHTML = '<i class="fas fa-upload"></i> 导入';
            }
        }
    };
    
    reader.readAsText(selectedLicenseFile);
}

// 点击模态窗口外部关闭
document.addEventListener('click', function(event) {
    const modal = document.getElementById('activation-modal');
    if (event.target === modal) {
        closeActivationModal();
    }
});

// AI助手功能
let aiSessionId = 'default';

function switchAITab(tab) {
    // 隐藏所有标签页内容
    document.querySelectorAll('.ai-tab-content').forEach(content => {
        content.style.display = 'none';
    });
    
    // 移除所有按钮的active类
    const tabButtons = document.querySelectorAll('#page-ai .tab-btn');
    tabButtons.forEach(btn => {
        btn.classList.remove('active');
    });
    
    // 显示选中的标签页
    const targetContent = document.getElementById(`ai-tab-${tab}`);
    if (targetContent) {
        targetContent.style.display = 'block';
    }
    
    // 激活对应的按钮
    if (tab === 'chat' && tabButtons[0]) {
        tabButtons[0].classList.add('active');
    } else if (tab === 'config' && tabButtons[1]) {
        tabButtons[1].classList.add('active');
    } else if (tab === 'server' && tabButtons[2]) {
        tabButtons[2].classList.add('active');
    }
}

function loadAIConfig() {
    fetch('/api/ai/config')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const config = data.config;
                document.getElementById('ai-api-url').value = config.api_url || 'https://api.siliconflow.cn/v1/chat/completions';
                document.getElementById('ai-api-key').value = config.api_key === '***' ? '' : config.api_key;
                document.getElementById('ai-model').value = config.model || 'deepseek-ai/DeepSeek-V3';
                document.getElementById('ai-timeout').value = config.timeout || 60;
            }
        })
        .catch(err => {
            console.error('加载AI配置失败:', err);
        });
}

function saveAIConfig() {
    const config = {
        api_url: document.getElementById('ai-api-url').value.trim(),
        api_key: document.getElementById('ai-api-key').value.trim(),
        model: document.getElementById('ai-model').value.trim(),
        timeout: parseInt(document.getElementById('ai-timeout').value) || 60
    };
    
    if (!config.api_url || !config.api_key || !config.model) {
        alert('请填写完整的配置信息');
        return;
    }
    
    fetch('/api/ai/config', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify(config)
    })
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                alert('配置已保存成功！');
            } else {
                alert('保存失败: ' + data.error);
            }
        })
        .catch(err => {
            alert('保存失败: ' + err.message);
        });
}

function resetAIConfig() {
    document.getElementById('ai-api-url').value = 'https://api.siliconflow.cn/v1/chat/completions';
    document.getElementById('ai-api-key').value = '';
    document.getElementById('ai-model').value = 'deepseek-ai/DeepSeek-V3';
    document.getElementById('ai-timeout').value = 60;
    alert('已恢复默认配置，请重新填入API密钥');
}

function toggleAIKeyVisibility() {
    const keyInput = document.getElementById('ai-api-key');
    const btn = event.target.closest('button');
    if (keyInput.type === 'password') {
        keyInput.type = 'text';
        btn.innerHTML = '<i class="fas fa-eye-slash"></i>';
    } else {
        keyInput.type = 'password';
        btn.innerHTML = '<i class="fas fa-eye"></i>';
    }
}

// 用于跟踪是否已经询问过执行命令（避免重复弹窗）
let aiCommandAsked = false;

function sendAIMessage() {
    const input = document.getElementById('ai-input-text');
    const message = input.value.trim();
    
    if (!message) {
        return;
    }
    
    // 重置命令询问标记
    aiCommandAsked = false;
    
    // 显示用户消息
    appendAIChat(`\n👤 您: ${message}\n`, 'user');
    input.value = '';
    
    // 发送到服务器
    fetch('/api/ai/chat', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({
            message: message,
            session_id: aiSessionId
        })
    })
        .then(res => {
            if (res.status === 403) {
                return res.json().then(data => {
                    appendAIChat(`\n❌ ${data.error}\n`, 'error');
                    if (data.need_activation) {
                        appendAIChat(`\n💡 请前往"关于"页面激活授权\n`, 'error');
                    }
                });
            }
            return res.json();
        })
        .then(data => {
            if (data && data.success) {
                appendAIChat(`\n🤖 AI助手: ${data.response}\n`, 'ai');
                
                // 解析AI响应中的命令（从```bash代码块中提取）
                parseAndAskExecuteCommand(data.response);
            } else if (data && !data.success) {
                appendAIChat(`\n❌ 错误: ${data.error}\n`, 'error');
            }
        })
        .catch(err => {
            appendAIChat(`\n❌ 请求失败: ${err.message}\n`, 'error');
        });
}

function parseAndAskExecuteCommand(aiResponse) {
    // 如果已经询问过，不再询问
    if (aiCommandAsked) {
        return;
    }
    
    // 查找代码块中的命令（支持bash、cmd、sh等）
    const cmdPattern = /```(?:bash|cmd|sh|powershell)?\n?([^`]+)```/g;
    const commands = [];
    let match;
    
    while ((match = cmdPattern.exec(aiResponse)) !== null) {
        const cmd = match[1].trim();
        if (cmd && !cmd.startsWith('#') && !cmd.startsWith('//')) {
            commands.push(cmd);
        }
    }
    
    // 只处理第一个命令，避免重复弹窗
    if (commands.length > 0 && !aiCommandAsked) {
        aiCommandAsked = true;
        const cmd = commands[0];
        
        // 检查危险命令
        const dangerousKeywords = ['rm -rf', 'rm -r', 'format', 'mkfs', 'dd if=', 'fdisk', 'chmod 777', 'chmod 000'];
        const isDangerous = dangerousKeywords.some(keyword => cmd.toLowerCase().includes(keyword));
        
        if (isDangerous) {
            appendAIChat(`\n⛔ 危险命令已阻止: ${cmd}\n\n`, 'error');
            return;
        }
        
        // 询问用户是否执行
        appendAIChat(`\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n`, 'system');
        appendAIChat(`🔧 检测到命令，是否执行？\n\n`, 'system');
        appendAIChat(`${cmd}\n\n`, 'command');
        
        if (confirm(`是否执行以下命令？\n\n${cmd}\n\n⚠️ 请确认命令安全后再执行`)) {
            appendAIChat(`▶️ 正在执行: ${cmd}\n`, 'system');
            executeAIServerCommand(cmd);
        } else {
            appendAIChat(`⏸️ 已取消执行: ${cmd}\n\n`, 'system');
        }
    }
}

function appendAIChat(message, type = 'normal') {
    const display = document.getElementById('ai-chat-display');
    if (!display) return;
    
    const div = document.createElement('div');
    div.style.marginBottom = '10px';
    div.style.lineHeight = '1.8';
    div.style.whiteSpace = 'pre-wrap'; // 保留换行和空格
    
    if (type === 'user') {
        div.style.color = '#1976d2';
        div.style.fontWeight = 'bold';
    } else if (type === 'ai') {
        div.style.color = '#388e3c';
        div.style.fontWeight = 'bold';
    } else if (type === 'error') {
        div.style.color = '#f44336';
    } else if (type === 'system') {
        div.style.color = '#909399';
        div.style.fontSize = '13px';
    } else if (type === 'command') {
        div.style.color = '#606266';
        div.style.fontFamily = 'monospace';
        div.style.background = '#f5f5f5';
        div.style.padding = '8px';
        div.style.borderRadius = '4px';
    } else if (type === 'result') {
        div.style.color = '#303133';
        div.style.fontFamily = 'monospace';
        div.style.background = '#f5f5f5';
        div.style.padding = '8px';
        div.style.borderRadius = '4px';
    } else {
        div.style.color = '#606266';
    }
    
    div.textContent = message;
    display.appendChild(div);
    display.scrollTop = display.scrollHeight;
}

function clearAIChat() {
    const display = document.getElementById('ai-chat-display');
    display.innerHTML = `
        <div style="color: #388e3c; font-weight: bold; margin-bottom: 10px;">🤖 欢迎使用AI助手！</div>
        <div style="color: #606266; line-height: 1.8;">
            <p>💡 我可以帮您：</p>
            <ul style="margin-left: 20px; padding-left: 0;">
                <li>工具功能："ping测试" "端口扫描" "DHCP检测" "环路检测"</li>
                <li>系统操作："查看IP配置" "查看系统负载" "查看进程"</li>
                <li>网络诊断：解决网络问题、分析网络流量</li>
            </ul>
            <p style="margin-top: 10px;">⚠️ 首次使用请先到"配置"页设置API密钥！</p>
            <p>请在下方输入您的需求... (回车发送，Shift+回车换行)</p>
        </div>
    `;
    
    fetch('/api/ai/clear', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({session_id: aiSessionId})
    });
}

function quickAIQuestion(question) {
    document.getElementById('ai-input-text').value = question;
    sendAIMessage();
}

// AI数据包分析
let aiAnalysisData = null; // 存储分析结果

function analyzePacketsWithAI() {
    console.log('AI分析按钮被点击');
    
    try {
    // 先检查授权
        checkAILicense().then(() => {
            console.log('授权检查通过');
            // 授权检查通过，获取全部数据包数据（用于AI分析）
            fetch('/api/packet/all')
                .then(res => {
                    console.log('获取全部数据包响应:', res.status);
                    return res.json();
                })
                .then(data => {
                    console.log('数据包数据:', data);
                    if (!data.success) {
                        alert('获取抓包状态失败: ' + (data.error || '未知错误'));
        return;
                    }
    
                    const packets = data.packets || [];
                    const totalCount = data.total_count !== undefined ? data.total_count : packets.length;
                    console.log('实际数据包数量:', totalCount, '返回的数据包数量:', packets.length);
    
                    if (totalCount === 0 || packets.length === 0) {
        alert('没有数据包可分析，请先进行抓包操作');
        return;
    }
    
                    if (!confirm(`是否使用AI分析 ${totalCount} 个数据包？`)) {
        return;
    }
    
                    // 打开AI分析模态窗口
                    openAIAnalysisModal();
                    
                    // 重置状态
                    aiAnalysisData = null;
                    showAIAnalysisProgress('正在准备分析数据...');
                    updateAIProgressBar(10);
                    
                    // 开始分析（使用全部数据包）
                    setTimeout(() => {
                        analyzePacketsWithAICore(packets);
                    }, 300);
                })
                .catch(err => {
                    console.error('获取数据包失败:', err);
                    alert('获取数据包失败: ' + err.message);
                });
        }).catch((err) => {
            console.error('授权检查失败:', err);
            alert('AI功能需要授权才能使用，请先激活授权');
        });
    } catch (err) {
        console.error('AI分析函数执行错误:', err);
        alert('执行AI分析时出错: ' + err.message);
    }
}

function analyzePacketsWithAICore(packets) {
    updateAIProgressBar(20);
    showAIAnalysisProgress('正在生成数据包摘要...');
    
    fetch('/api/ai/analyze/packets', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({packets: packets})
    })
        .then(res => {
            if (res.status === 403) {
                return res.json().then(data => {
                    showAIAnalysisError(`❌ ${data.error}\n\n💡 请前往"关于"页面激活授权`);
                });
            }
            updateAIProgressBar(50);
            showAIAnalysisProgress('正在调用AI大模型分析...');
            return res.json();
        })
        .then(data => {
            if (data && data.success) {
                updateAIProgressBar(90);
                showAIAnalysisProgress('分析完成，正在格式化结果...');
                
                // 保存分析结果
                aiAnalysisData = {
                    analysis: data.analysis,
                    summary: data.summary,
                    timestamp: new Date().toLocaleString('zh-CN')
                };
                
                setTimeout(() => {
                    updateAIProgressBar(100);
                    showAIAnalysisResult(data.analysis);
                }, 500);
            } else if (data && !data.success) {
                showAIAnalysisError(`❌ AI分析失败: ${data.error}`);
            }
        })
        .catch(err => {
            showAIAnalysisError(`❌ 请求失败: ${err.message}\n\n可能原因：\n1. 网络连接不稳定\n2. AI服务响应较慢\n3. 数据量过大\n\n建议：\n- 检查网络连接\n- 稍后再试\n- 减少分析的数据包数量`);
        });
}

// 打开AI分析模态窗口
function openAIAnalysisModal() {
    const modal = document.getElementById('ai-analysis-modal');
    if (modal) {
        modal.style.display = 'flex';
        modal.classList.add('show');
        // 重置显示状态
        const progress = document.getElementById('ai-analysis-progress');
        const result = document.getElementById('ai-analysis-result');
        const error = document.getElementById('ai-analysis-error');
        
        if (progress) progress.style.display = 'block';
        if (result) result.style.display = 'none';
        if (error) error.style.display = 'none';
        
        console.log('AI分析模态窗口已打开');
    } else {
        console.error('找不到AI分析模态窗口元素');
    }
}

// 关闭AI分析模态窗口
function closeAIAnalysisModal() {
    const modal = document.getElementById('ai-analysis-modal');
    if (modal) {
        modal.style.display = 'none';
        modal.classList.remove('show');
    }
}

// 显示分析进度
function showAIAnalysisProgress(text) {
    const progressText = document.getElementById('ai-progress-text');
    if (progressText) {
        progressText.textContent = text;
    }
}

// 更新进度条
function updateAIProgressBar(percent) {
    const progressBar = document.getElementById('ai-progress-bar');
    if (progressBar) {
        progressBar.style.width = percent + '%';
    }
}

// 显示分析结果
function showAIAnalysisResult(analysis) {
    document.getElementById('ai-analysis-progress').style.display = 'none';
    document.getElementById('ai-analysis-result').style.display = 'block';
    document.getElementById('ai-analysis-error').style.display = 'none';
    
    const content = document.getElementById('ai-analysis-content');
    if (content) {
        // 格式化显示（支持Markdown样式）
        content.textContent = analysis;
        // 可以在这里添加Markdown渲染
    }
}

// 显示分析错误
function showAIAnalysisError(errorMsg) {
    document.getElementById('ai-analysis-progress').style.display = 'none';
    document.getElementById('ai-analysis-result').style.display = 'none';
    document.getElementById('ai-analysis-error').style.display = 'block';
    
    const errorMessage = document.getElementById('ai-error-message');
    if (errorMessage) {
        errorMessage.textContent = errorMsg;
    }
}

// 复制AI分析报告
function copyAIAnalysisReport() {
    if (!aiAnalysisData) {
        alert('没有可复制的分析报告');
        return;
    }
    
    const report = `AI大模型网络流量分析报告\n生成时间: ${aiAnalysisData.timestamp}\n${'='.repeat(60)}\n\n${aiAnalysisData.analysis}\n\n${'='.repeat(60)}\n\n数据包摘要:\n${aiAnalysisData.summary}`;
    
    // 复制到剪贴板
    if (navigator.clipboard && navigator.clipboard.writeText) {
        navigator.clipboard.writeText(report).then(() => {
            const btn = document.getElementById('copy-ai-report-btn');
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-check"></i> 已复制';
            btn.style.background = '#4CAF50';
            setTimeout(() => {
                btn.innerHTML = originalText;
                btn.style.background = '';
            }, 2000);
        }).catch(err => {
            alert('复制失败: ' + err.message);
        });
    } else {
        // 备用方案
        const textarea = document.createElement('textarea');
        textarea.value = report;
        document.body.appendChild(textarea);
        textarea.select();
        try {
            document.execCommand('copy');
            alert('报告已复制到剪贴板');
        } catch (err) {
            alert('复制失败，请手动复制');
        }
        document.body.removeChild(textarea);
    }
}

// 保存AI分析报告
function saveAIAnalysisReport() {
    if (!aiAnalysisData) {
        alert('没有可保存的分析报告');
        return;
    }
    
    const report = `AI大模型网络流量分析报告\n生成时间: ${aiAnalysisData.timestamp}\n${'='.repeat(60)}\n\n${aiAnalysisData.analysis}\n\n${'='.repeat(60)}\n\n数据包摘要:\n${aiAnalysisData.summary}`;
    
    // 创建下载链接
    const blob = new Blob([report], { type: 'text/plain;charset=utf-8' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `AI流量分析报告_${new Date().toISOString().slice(0, 19).replace(/:/g, '-')}.txt`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    const btn = document.getElementById('save-ai-report-btn');
    const originalText = btn.innerHTML;
    btn.innerHTML = '<i class="fas fa-check"></i> 已保存';
    btn.style.background = '#4CAF50';
    setTimeout(() => {
        btn.innerHTML = originalText;
        btn.style.background = '';
    }, 2000);
}

// 点击模态窗口外部关闭
document.addEventListener('click', function(event) {
    const modal = document.getElementById('ai-analysis-modal');
    if (event.target === modal) {
        closeAIAnalysisModal();
    }
});

// AI服务器配置
function executeAIServerTask() {
    // 先检查授权
    checkAILicense().catch(() => {
        alert('AI功能需要授权才能使用，请先激活授权');
        return;
    });
    
    const taskInput = document.getElementById('ai-server-task');
    const task = taskInput ? taskInput.value.trim() : '';
    const resultDiv = document.getElementById('ai-server-result');
    
    if (!task) {
        alert('请输入任务描述');
        return;
    }
    
    if (resultDiv) {
        resultDiv.textContent = '正在执行任务，请稍候...';
    }
    
    // 全自动化执行，不需要确认
    fetch('/api/ai/server/execute', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({task: task})
    })
        .then(res => {
            if (res.status === 403) {
                return res.json().then(data => {
                    if (resultDiv) {
                        resultDiv.textContent = `❌ ${data.error}\n\n💡 请前往"关于"页面激活授权`;
                    }
                });
            }
            return res.json();
        })
        .then(data => {
            if (data && data.success) {
                const result = data.result;
                let output = '=== 任务执行完成 ===\n\n';
                
                if (result && result.results && Array.isArray(result.results)) {
                    result.results.forEach((item, index) => {
                        output += `步骤 ${index + 1}: ${item.step || item.description || '执行命令'}\n`;
                        output += `命令: ${item.command || 'N/A'}\n`;
                        if (item.result) {
                            if (item.result.success) {
                                output += `结果:\n${item.result.stdout || item.result.message || '执行成功'}\n`;
                                if (item.result.stderr) {
                                    output += `警告: ${item.result.stderr}\n`;
                                }
                            } else {
                                output += `错误: ${item.result.error || item.result.stderr || '执行失败'}\n`;
                            }
                        }
                        output += '\n';
                    });
                } else if (result && result.message) {
                    output += result.message + '\n';
                    if (result.stdout) {
                        output += `输出:\n${result.stdout}\n`;
                    }
                } else if (result) {
                    // 如果是单个命令结果
                    if (result.success) {
                        output += `执行成功\n`;
                        if (result.stdout) {
                            output += `输出:\n${result.stdout}\n`;
                        }
                } else {
                        output += `执行失败: ${result.error || result.stderr || '未知错误'}\n`;
                    }
                } else {
                    output += '任务执行成功\n';
                }
                
                if (resultDiv) {
                    resultDiv.textContent = output;
                }
            } else if (data && !data.success) {
                if (resultDiv) {
                    resultDiv.textContent = `❌ 执行失败: ${data.error || '未知错误'}`;
                }
            }
        })
        .catch(err => {
            if (resultDiv) {
                resultDiv.textContent = `❌ 请求失败: ${err.message}`;
            }
        });
}

function clearAIServerResult() {
    const resultDiv = document.getElementById('ai-server-result');
    const taskInput = document.getElementById('ai-server-task');
    if (resultDiv) {
        resultDiv.textContent = '等待执行任务...';
    }
    if (taskInput) {
        taskInput.value = '';
    }
}

// Web终端功能
let terminal = null;
let terminalConnected = false;
let terminalProcess = null;

function initTerminal() {
    if (terminal) {
        return; // 已经初始化
    }
    
    const terminalElement = document.getElementById('terminal');
    if (!terminalElement) {
        return;
    }
    
    // 初始化xterm终端
    terminal = new Terminal({
        cursorBlink: true,
        fontSize: 14,
        fontFamily: 'Consolas, "Courier New", monospace',
        theme: {
            background: '#000000',
            foreground: '#00ff00',
            cursor: '#00ff00',
            selection: '#ffffff33'
        }
    });
    
    const fitAddon = new FitAddon.FitAddon();
    terminal.loadAddon(fitAddon);
    
    terminal.open(terminalElement);
    fitAddon.fit();
    
    // 监听窗口大小变化
    window.addEventListener('resize', () => {
        if (terminal && fitAddon) {
            fitAddon.fit();
        }
    });
    
    // 监听终端输入
    terminal.onData((data) => {
        if (terminalConnected && terminalProcess) {
            sendTerminalInput(data);
        } else {
            // 如果未连接，显示提示
            terminal.write(data);
        }
    });
    
    // 显示欢迎信息
    terminal.writeln('\r\n\x1b[32m欢迎使用Web终端\x1b[0m');
    terminal.writeln('提示：点击"连接终端"按钮开始使用\r\n');
    terminal.write('$ ');
}

function connectTerminal() {
    if (terminalConnected) {
        return;
    }
    
    if (!terminal) {
        initTerminal();
    }
    
    // 检查授权
    checkAILicense().then(() => {
        terminal.clear();
        terminal.writeln('\r\n\x1b[33m正在连接终端...\x1b[0m\r\n');
        
        // 启动终端会话
        fetch('/api/terminal/start', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'}
        })
            .then(res => {
                if (res.status === 403) {
                    terminal.writeln('\r\n\x1b[31m错误: 需要授权才能使用此功能\x1b[0m');
                    terminal.writeln('请前往"关于"页面激活授权\r\n');
                    return null;
                }
                return res.json();
            })
            .then(data => {
                if (data && data.success) {
                    terminalProcess = data.session_id;
                    terminalConnected = true;
                    
                    document.getElementById('terminal-connect-btn').disabled = true;
                    document.getElementById('terminal-disconnect-btn').disabled = false;
                    
                    terminal.writeln('\r\n\x1b[32m✓ 终端已连接\x1b[0m\r\n');
                    terminal.write('$ ');
                    
                    // 开始轮询输出
                    pollTerminalOutput();
                } else if (data) {
                    terminal.writeln(`\r\n\x1b[31m连接失败: ${data.error || '未知错误'}\x1b[0m\r\n`);
                }
            })
            .catch(err => {
                terminal.writeln(`\r\n\x1b[31m连接失败: ${err.message}\x1b[0m\r\n`);
            });
    }).catch(() => {
        if (terminal) {
            terminal.writeln('\r\n\x1b[31m错误: 需要授权才能使用此功能\x1b[0m');
            terminal.writeln('请前往"关于"页面激活授权\r\n');
        } else {
            alert('需要授权才能使用此功能，请先激活授权');
        }
    });
}

function disconnectTerminal() {
    if (!terminalConnected) {
        return;
    }
    
    if (terminalProcess) {
        fetch('/api/terminal/stop', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({session_id: terminalProcess})
        }).catch(() => {});
    }
    
    terminalConnected = false;
    terminalProcess = null;
    
    document.getElementById('terminal-connect-btn').disabled = false;
    document.getElementById('terminal-disconnect-btn').disabled = true;
    
    if (terminal) {
        terminal.writeln('\r\n\x1b[33m终端已断开连接\x1b[0m\r\n');
    }
}

function sendTerminalInput(data) {
    if (!terminalProcess || !terminalConnected) {
        return;
    }
    
    fetch('/api/terminal/input', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({
            session_id: terminalProcess,
            input: data
        })
    }).catch(err => {
        console.error('发送输入失败:', err);
    });
}

function pollTerminalOutput() {
    if (!terminalConnected || !terminalProcess) {
        return;
    }
    
    fetch('/api/terminal/output', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({session_id: terminalProcess})
    })
        .then(res => res.json())
        .then(data => {
            if (data && data.success && data.output) {
                if (terminal) {
                    terminal.write(data.output);
                }
            }
            
            // 继续轮询
            if (terminalConnected) {
                setTimeout(pollTerminalOutput, 100);
            }
        })
        .catch(err => {
            console.error('获取输出失败:', err);
            if (terminalConnected) {
                setTimeout(pollTerminalOutput, 1000);
            }
        });
}

function clearTerminal() {
    if (terminal) {
        terminal.clear();
        if (terminalConnected) {
            terminal.write('$ ');
        }
    }
}

// 页面加载时初始化终端（延迟执行）
setTimeout(() => {
    const serverTab = document.getElementById('ai-tab-server');
    if (serverTab) {
        const observer = new MutationObserver((mutations) => {
            if (serverTab.style.display !== 'none') {
                initTerminal();
}
        });
        observer.observe(serverTab, { attributes: true, attributeFilter: ['style'] });
    }
}, 1000);

function executeAIServerCommand(command) {
    // 在AI对话中执行命令，显示在对话区域
    appendAIChat(`\n📋 命令输出:\n`, 'system');
    
    fetch('/api/ai/server/execute', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({command: command})
    })
        .then(res => {
            if (res.status === 403) {
                return res.json().then(data => {
                    appendAIChat(`❌ ${data.error}\n\n💡 请前往"关于"页面激活授权\n\n`, 'error');
                });
            }
            return res.json();
        })
        .then(data => {
            if (data && data.success) {
                const result = data.result;
                let output = '';
                
                if (result) {
                    if (result.stdout) {
                        output = result.stdout;
                    } else if (result.message) {
                        output = result.message;
                    } else if (result.success) {
                        output = '命令执行成功';
                    } else if (result.error) {
                        output = `错误: ${result.error}`;
                    } else if (result.stderr) {
                        output = `警告: ${result.stderr}`;
            } else {
                        output = '命令执行完成，无输出';
                    }
                } else {
                    output = '命令执行完成，无输出';
                }
                
                if (output) {
                    appendAIChat(`${output}\n\n`, 'result');
                } else {
                    appendAIChat(`命令执行完成\n\n`, 'system');
                }
            } else {
                const error = data && data.error ? data.error : '执行失败';
                appendAIChat(`❌ 错误: ${error}\n\n`, 'error');
            }
        })
        .catch(err => {
            appendAIChat(`❌ 请求失败: ${err.message}\n\n`, 'error');
        });
}

// 回车发送消息
document.addEventListener('DOMContentLoaded', function() {
    const aiInput = document.getElementById('ai-input-text');
    if (aiInput) {
        aiInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendAIMessage();
            }
        });
    }
    
    // 初始化连接测试页面（延迟执行，避免阻塞页面加载）
    // 只在切换到该页面时才初始化，不在DOMContentLoaded时初始化
});

// ==================== 连接测试功能 ====================

function switchToolsTab(tab) {
    const pageTools = document.getElementById('page-networktools');
    if (!pageTools) return; // 如果页面不存在，直接返回
    
    // 隐藏所有标签页内容
    const tabContents = pageTools.querySelectorAll('.tools-tab-content');
    tabContents.forEach(el => {
        el.style.display = 'none';
        el.classList.remove('active');
    });
    
    // 重置所有按钮样式
    const buttons = pageTools.querySelectorAll('.tab-btn');
    buttons.forEach(btn => {
        btn.classList.remove('active');
    });
    
    // 显示选中的标签页
    const targetTab = document.getElementById(`tools-tab-${tab}`);
    if (targetTab) {
        targetTab.style.display = 'block';
        targetTab.classList.add('active');
    }
    
    // 高亮选中的按钮 - 使用data属性匹配（更高效）
    buttons.forEach(btn => {
        const btnText = btn.textContent.trim();
        const tabNames = {
            'dns': 'DNS查询',
            'dns_resolve': 'DNS解析测试',
            'connectivity': '站点测试',
            'mtu': 'MTU测试',
            'ssl': 'SSL证书检查',
            'password': '密码生成',
            'connection_stats': '连接统计'
        };
        if (btnText === tabNames[tab]) {
            btn.classList.add('active');
        }
    });
    
    // 如果是连接统计，自动刷新（延迟执行避免阻塞）
    if (tab === 'connection_stats') {
        setTimeout(() => {
            if (document.getElementById('tools-connection-stats-result')) {
                handleConnectionStats();
            }
        }, 100);
    }
}

function handleDNSLookup() {
    const domain = document.getElementById('tools-dns-domain').value.trim();
    const recordType = document.getElementById('tools-dns-type').value;
    const result = document.getElementById('tools-dns-result');
    
    if (!domain) {
        result.textContent = '❌ 请输入域名';
        return;
    }
    
    result.textContent = '正在查询...';
    
    fetch('/api/tools/dns/lookup', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({domain, record_type: recordType})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            let output = `DNS查询结果 - ${data.domain}\n`;
            output += '='.repeat(50) + '\n\n';
            
            if (data.records && data.records.length > 0) {
                data.records.forEach(record => {
                    output += `📍 ${record.type}记录:\n`;
                    output += `  • ${record.value}\n`;
                    if (record.ttl && record.ttl !== 'N/A') {
                        output += `  • TTL: ${record.ttl}\n`;
                    }
                    output += '\n';
                });
            } else {
                output += '未找到记录\n';
            }
            
            result.textContent = output;
        } else {
            result.textContent = `❌ 查询失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

function handleDNSResolve() {
    const domain = document.getElementById('tools-dns-resolve-domain').value.trim();
    const result = document.getElementById('tools-dns-resolve-result');
    
    if (!domain) {
        result.textContent = '❌ 请输入域名';
        return;
    }
    
    result.textContent = '正在测试DNS解析速度...';
    
    fetch('/api/tools/dns/resolve', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({domain})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            let output = `DNS解析测试结果 - ${data.domain}\n`;
            output += '='.repeat(60) + '\n\n';
            
            if (data.dns_servers && data.dns_servers.length > 0) {
                data.dns_servers.forEach(server => {
                    if (server.success) {
                        output += `✅ ${server.name} (${server.ip}):\n`;
                        output += `   响应时间: ${server.response_time}ms\n`;
                        output += `   解析IP: ${server.resolved_ip}\n\n`;
                    } else {
                        output += `❌ ${server.name} (${server.ip}):\n`;
                        output += `   错误: ${server.error || '解析失败'}\n\n`;
                    }
                });
            }
            
            result.textContent = output;
        } else {
            result.textContent = `❌ 测试失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

function handleConnectivityTest() {
    const result = document.getElementById('tools-connectivity-result');
    result.textContent = '正在测试互联网连接...';
    
    fetch('/api/tools/connectivity/test', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'}
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            let output = '互联网连接测试结果\n';
            output += '='.repeat(60) + '\n\n';
            
            if (data.tests && data.tests.length > 0) {
                data.tests.forEach(test => {
                    if (test.success) {
                        output += `✅ ${test.name} (${test.host}:${test.port}):\n`;
                        output += `   响应时间: ${test.response_time}ms\n\n`;
                    } else {
                        output += `❌ ${test.name} (${test.host}:${test.port}):\n`;
                        output += `   错误: ${test.error || '连接失败'}\n\n`;
                    }
                });
            }
            
            result.textContent = output;
        } else {
            result.textContent = `❌ 测试失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

function handleMTUTest() {
    const host = document.getElementById('tools-mtu-host').value.trim();
    const result = document.getElementById('tools-mtu-result');
    
    if (!host) {
        result.textContent = '❌ 请输入目标主机';
        return;
    }
    
    result.textContent = '正在测试MTU...';
    
    fetch('/api/tools/mtu/test', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({host})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            let output = `MTU测试结果 - ${data.host}\n`;
            output += '='.repeat(60) + '\n\n';
            output += `最大传输单元 (MTU): ${data.mtu || '未知'} 字节\n\n`;
            
            if (data.details && data.details.length > 0) {
                output += '测试详情:\n';
                data.details.forEach(detail => {
                    if (detail.success) {
                        output += `✅ 包大小 ${detail.size} 字节 (总大小 ${detail.total_size} 字节): 响应时间 ${detail.response_time}ms\n`;
                    } else {
                        output += `❌ 包大小 ${detail.size} 字节: ${detail.error || '失败'}\n`;
                    }
                });
            }
            
            result.textContent = output;
        } else {
            result.textContent = `❌ 测试失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

function handleSSLCheck() {
    const domain = document.getElementById('tools-ssl-domain').value.trim();
    const result = document.getElementById('tools-ssl-result');
    
    if (!domain) {
        result.textContent = '❌ 请输入域名或URL';
        return;
    }
    
    result.textContent = '正在检查SSL证书...';
    
    fetch('/api/tools/ssl/check', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({domain})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success && data.cert_info) {
            const cert = data.cert_info;
            let output = `SSL证书检查结果 - ${data.domain}\n`;
            output += '='.repeat(60) + '\n\n';
            output += `主题: ${cert.subject?.CN || cert.subject?.commonName || JSON.stringify(cert.subject) || 'N/A'}\n`;
            output += `颁发者: ${cert.issuer?.CN || cert.issuer?.commonName || JSON.stringify(cert.issuer) || 'N/A'}\n`;
            output += `有效期开始: ${cert.not_before || 'N/A'}\n`;
            output += `有效期结束: ${cert.not_after || 'N/A'}\n`;
            output += `剩余天数: ${cert.days_left || 'N/A'} 天\n`;
            output += `证书状态: ${cert.is_valid ? '✅ 有效' : '❌ 已过期'}\n`;
            
            if (cert.subject_alt_name && cert.subject_alt_name.length > 0) {
                output += `\n备用名称 (SAN):\n`;
                cert.subject_alt_name.forEach(san => {
                    if (Array.isArray(san) && san.length >= 2) {
                        output += `  • ${san[1]}\n`;
                    } else {
                        output += `  • ${san}\n`;
                    }
                });
            }
            
            result.textContent = output;
        } else {
            result.textContent = `❌ 检查失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

function handlePasswordGenerate() {
    const length = parseInt(document.getElementById('tools-password-length').value) || 16;
    const uppercase = document.getElementById('tools-password-uppercase').checked;
    const lowercase = document.getElementById('tools-password-lowercase').checked;
    const digits = document.getElementById('tools-password-digits').checked;
    const special = document.getElementById('tools-password-special').checked;
    const result = document.getElementById('tools-password-result');
    
    if (!uppercase && !lowercase && !digits && !special) {
        result.textContent = '❌ 请至少选择一种字符类型';
        return;
    }
    
    result.textContent = '正在生成密码...';
    
    fetch('/api/tools/password/generate', {
        method: 'POST',
        headers: {'Content-Type': 'application/json'},
        body: JSON.stringify({length, uppercase, lowercase, digits, special})
    })
    .then(res => res.json())
    .then(data => {
        if (data.success) {
            result.textContent = data.password;
        } else {
            result.textContent = `❌ 生成失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

function copyPassword() {
    const result = document.getElementById('tools-password-result');
    const password = result.textContent.trim();
    
    if (!password || password.startsWith('❌')) {
        alert('请先生成密码');
        return;
    }
    
    navigator.clipboard.writeText(password).then(() => {
        alert('密码已复制到剪贴板');
    }).catch(() => {
        // 备用方案
        const textarea = document.createElement('textarea');
        textarea.value = password;
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
        alert('密码已复制到剪贴板');
    });
}

function handleConnectionStats() {
    const result = document.getElementById('tools-connection-stats-result');
    result.textContent = '正在获取连接统计...';
    
    fetch('/api/tools/connection/stats')
    .then(res => res.json())
    .then(data => {
        if (data.success && data.stats) {
            const stats = data.stats;
            let output = '网络连接统计\n';
            output += '='.repeat(60) + '\n\n';
            output += `总连接数: ${stats.total || 0}\n`;
            output += `TCP连接: ${stats.tcp || 0}\n`;
            output += `UDP连接: ${stats.udp || 0}\n`;
            output += `已建立连接: ${stats.established || 0}\n`;
            output += `监听连接: ${stats.listen || 0}\n`;
            output += `TIME_WAIT: ${stats.time_wait || 0}\n`;
            output += `CLOSE_WAIT: ${stats.close_wait || 0}\n`;
            output += `其他状态: ${stats.other || 0}\n\n`;
            
            if (stats.by_state && Object.keys(stats.by_state).length > 0) {
                output += '按状态分类:\n';
                for (const [state, count] of Object.entries(stats.by_state)) {
                    output += `  ${state}: ${count}\n`;
                }
            }
            
            result.textContent = output;
        } else {
            result.textContent = `❌ 获取失败: ${data.error || '未知错误'}`;
        }
    })
    .catch(err => {
        result.textContent = `❌ 请求失败: ${err.message}`;
    });
}

// 检查AI授权
function checkAILicense() {
    return fetch('/api/license/status')
        .then(res => res.json())
        .then(data => {
            if (data.success && data.is_activated) {
                return Promise.resolve();
            } else {
                return Promise.reject(new Error('未授权'));
            }
        })
        .catch(err => {
            console.error('检查授权失败:', err);
            return Promise.reject(err);
        });
}

// 导出所有连接测试相关函数到全局作用域，供HTML onclick使用
window.switchToolsTab = switchToolsTab;
window.handleDNSLookup = handleDNSLookup;
window.handleDNSResolve = handleDNSResolve;
window.handleConnectivityTest = handleConnectivityTest;
window.handleMTUTest = handleMTUTest;
window.handleSSLCheck = handleSSLCheck;
window.handlePasswordGenerate = handlePasswordGenerate;
window.copyPassword = copyPassword;
window.handleConnectionStats = handleConnectionStats;
