#!/bin/bash
# 网络测试工具箱 - 一键安装脚本
# 支持: Ubuntu/Debian, CentOS/RHEL, 麒麟/UOS/统信

set -e

INSTALL_DIR="/opt/network_toolkit"
SERVICE_NAME="network-toolkit"
PYTHON_VERSION="3.12"
PYTHON_CMD=""

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

log_info() { echo -e "${GREEN}[INFO]${NC} $1"; }
log_warn() { echo -e "${YELLOW}[WARN]${NC} $1"; }
log_error() { echo -e "${RED}[ERROR]${NC} $1"; }

echo "=========================================="
echo "  网络测试工具箱 - 安装程序"
echo "=========================================="
echo ""

# 检查root权限
if [ "$EUID" -ne 0 ]; then
    log_error "请使用 root 权限运行此脚本"
    exit 1
fi

# 检测系统类型
detect_os() {
    if [ -f /etc/os-release ]; then
        . /etc/os-release
        OS=$ID
        OS_VERSION=$VERSION_ID
        OS_NAME=$PRETTY_NAME
    elif [ -f /etc/redhat-release ]; then
        OS="centos"
        OS_VERSION=$(cat /etc/redhat-release | grep -oE '[0-9]+' | head -1)
    else
        OS="unknown"
    fi
    
    # 检测国产系统
    case "$OS" in
        kylin|openkylin|neokylin)
            OS_FAMILY="kylin"
            ;;
        uos|deepin|uniontech)
            OS_FAMILY="debian"
            ;;
        ubuntu|debian|linuxmint)
            OS_FAMILY="debian"
            ;;
        centos|rhel|fedora|rocky|almalinux|openeuler|anolis)
            OS_FAMILY="rhel"
            ;;
        *)
            # 尝试检测包管理器
            if command -v apt-get &> /dev/null; then
                OS_FAMILY="debian"
            elif command -v yum &> /dev/null || command -v dnf &> /dev/null; then
                OS_FAMILY="rhel"
            else
                OS_FAMILY="unknown"
            fi
            ;;
    esac
    
    log_info "检测到系统: $OS_NAME ($OS_FAMILY)"
}

# 安装Python 3.12
install_python312() {
    log_info "检查 Python 3.12..."
    
    # 检查是否已安装
    if command -v python3.12 &> /dev/null; then
        PYTHON_CMD="python3.12"
        log_info "Python 3.12 已安装"
        return 0
    fi
    
    log_info "安装 Python 3.12..."
    
    case "$OS_FAMILY" in
        debian)
            # Ubuntu/Debian/UOS/Deepin/麒麟
            apt-get update -qq
            
            # 尝试直接安装
            if apt-cache show python3.12 &> /dev/null; then
                apt-get install -y python3.12 python3.12-venv python3.12-dev -qq
            else
                # 添加deadsnakes PPA (Ubuntu)
                if [ "$OS" = "ubuntu" ]; then
                    apt-get install -y software-properties-common -qq
                    add-apt-repository -y ppa:deadsnakes/ppa
                    apt-get update -qq
                    apt-get install -y python3.12 python3.12-venv python3.12-dev -qq
                else
                    # 从源码编译
                    install_python_from_source
                    return $?
                fi
            fi
            PYTHON_CMD="python3.12"
            ;;
            
        rhel)
            # CentOS/RHEL/Rocky/openEuler/Anolis
            if command -v dnf &> /dev/null; then
                PKG_MGR="dnf"
            else
                PKG_MGR="yum"
            fi
            
            # 安装编译依赖
            $PKG_MGR groupinstall -y "Development Tools" -q 2>/dev/null || true
            $PKG_MGR install -y gcc openssl-devel bzip2-devel libffi-devel zlib-devel readline-devel sqlite-devel wget -q
            
            # 尝试从EPEL或AppStream安装
            if $PKG_MGR list python3.12 &> /dev/null 2>&1; then
                $PKG_MGR install -y python3.12 python3.12-devel -q
                PYTHON_CMD="python3.12"
            else
                # 从源码编译
                install_python_from_source
                return $?
            fi
            ;;
            
        kylin)
            # 麒麟系统
            if command -v apt-get &> /dev/null; then
                apt-get update -qq
                apt-get install -y build-essential libssl-dev zlib1g-dev libbz2-dev \
                    libreadline-dev libsqlite3-dev wget libffi-dev -qq
            elif command -v yum &> /dev/null; then
                yum install -y gcc openssl-devel bzip2-devel libffi-devel zlib-devel \
                    readline-devel sqlite-devel wget -q
            fi
            install_python_from_source
            return $?
            ;;
            
        *)
            log_warn "未知系统，尝试从源码编译Python"
            install_python_from_source
            return $?
            ;;
    esac
    
    return 0
}

# 从源码编译Python
install_python_from_source() {
    log_info "从源码编译 Python 3.12..."
    
    # 安装编译依赖
    if command -v apt-get &> /dev/null; then
        apt-get update -qq
        apt-get install -y build-essential libssl-dev zlib1g-dev libbz2-dev \
            libreadline-dev libsqlite3-dev wget libffi-dev libncurses5-dev \
            libgdbm-dev libnss3-dev liblzma-dev -qq
    elif command -v yum &> /dev/null || command -v dnf &> /dev/null; then
        PKG_MGR=$(command -v dnf &> /dev/null && echo "dnf" || echo "yum")
        $PKG_MGR groupinstall -y "Development Tools" -q 2>/dev/null || true
        $PKG_MGR install -y gcc openssl-devel bzip2-devel libffi-devel zlib-devel \
            readline-devel sqlite-devel wget ncurses-devel gdbm-devel xz-devel -q
    fi
    
    cd /tmp
    PYTHON_TAR="Python-3.12.0.tgz"
    
    if [ ! -f "$PYTHON_TAR" ]; then
        log_info "下载 Python 3.12.0..."
        wget -q https://www.python.org/ftp/python/3.12.0/Python-3.12.0.tgz || \
        wget -q https://mirrors.huaweicloud.com/python/3.12.0/Python-3.12.0.tgz || \
        wget -q https://npm.taobao.org/mirrors/python/3.12.0/Python-3.12.0.tgz
    fi
    
    tar -xzf Python-3.12.0.tgz
    cd Python-3.12.0
    
    log_info "配置编译选项..."
    ./configure --enable-optimizations --prefix=/usr/local/python312 --with-ensurepip=install
    
    log_info "编译中 (可能需要几分钟)..."
    make -j$(nproc)
    make altinstall
    
    # 创建软链接
    ln -sf /usr/local/python312/bin/python3.12 /usr/local/bin/python3.12
    ln -sf /usr/local/python312/bin/pip3.12 /usr/local/bin/pip3.12
    
    PYTHON_CMD="/usr/local/python312/bin/python3.12"
    
    # 清理
    cd /tmp
    rm -rf Python-3.12.0 Python-3.12.0.tgz
    
    log_info "Python 3.12 编译安装完成"
    return 0
}

# 安装系统依赖
install_dependencies() {
    log_info "安装系统依赖..."
    
    if command -v apt-get &> /dev/null; then
        apt-get update -qq
        apt-get install -y tcpdump traceroute net-tools iproute2 -qq 2>/dev/null || true
    elif command -v yum &> /dev/null || command -v dnf &> /dev/null; then
        PKG_MGR=$(command -v dnf &> /dev/null && echo "dnf" || echo "yum")
        $PKG_MGR install -y tcpdump traceroute net-tools iproute -q 2>/dev/null || true
    fi
}

# 主安装流程
main() {
    detect_os
    
    echo ""
    log_info "[1/6] 安装 Python 3.12 环境..."
    install_python312
    
    if [ -z "$PYTHON_CMD" ] || ! command -v $PYTHON_CMD &> /dev/null; then
        log_error "Python 3.12 安装失败"
        exit 1
    fi
    
    log_info "使用 Python: $($PYTHON_CMD --version)"
    
    log_info "[2/6] 安装系统依赖..."
    install_dependencies
    
    log_info "[3/6] 创建安装目录..."
    rm -rf "$INSTALL_DIR"
    mkdir -p "$INSTALL_DIR"
    
    SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
    cp -r "$SCRIPT_DIR"/* "$INSTALL_DIR/"
    rm -f "$INSTALL_DIR/install.sh"
    
    log_info "[4/6] 创建虚拟环境..."
    cd "$INSTALL_DIR"
    $PYTHON_CMD -m venv venv
    source venv/bin/activate
    pip install --upgrade pip -q
    pip install -r requirements.txt -q
    
    log_info "[5/6] 创建系统服务..."
    cat > /etc/systemd/system/${SERVICE_NAME}.service << EOF
[Unit]
Description=Network Toolkit Web Service
After=network.target

[Service]
Type=simple
User=root
WorkingDirectory=${INSTALL_DIR}
ExecStart=${INSTALL_DIR}/venv/bin/python ${INSTALL_DIR}/app.py
Restart=always
RestartSec=3
Environment=PYTHONUNBUFFERED=1

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload
    systemctl enable ${SERVICE_NAME} 2>/dev/null
    
    log_info "[6/6] 启动服务..."
    systemctl start ${SERVICE_NAME}
    
    sleep 2
    
    # 获取IP
    IP=$(hostname -I 2>/dev/null | awk '{print $1}')
    [ -z "$IP" ] && IP=$(ip route get 1 2>/dev/null | awk '{print $7}' | head -1)
    [ -z "$IP" ] && IP="localhost"
    
    echo ""
    echo "=========================================="
    echo -e "${GREEN}  安装完成!${NC}"
    echo "=========================================="
    echo "安装目录: $INSTALL_DIR"
    echo "Python: $($PYTHON_CMD --version)"
    echo ""
    echo -e "访问地址: ${GREEN}http://${IP}:5000${NC}"
    echo ""
    echo "管理命令:"
    echo "  启动: systemctl start ${SERVICE_NAME}"
    echo "  停止: systemctl stop ${SERVICE_NAME}"
    echo "  重启: systemctl restart ${SERVICE_NAME}"
    echo "  状态: systemctl status ${SERVICE_NAME}"
    echo "  日志: journalctl -u ${SERVICE_NAME} -f"
    echo "=========================================="
}

main "$@"
